"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CASAuthentication = void 0;
const app_logs_1 = require("./app.logs");
const url_1 = __importDefault(require("url"));
const http_1 = __importDefault(require("http"));
const https_1 = __importDefault(require("https"));
const xml2js_1 = __importDefault(require("xml2js"));
var AUTH_TYPE;
(function (AUTH_TYPE) {
    AUTH_TYPE[AUTH_TYPE["BOUNCE"] = 0] = "BOUNCE";
    AUTH_TYPE[AUTH_TYPE["BOUNCE_REDIRECT"] = 1] = "BOUNCE_REDIRECT";
    AUTH_TYPE[AUTH_TYPE["BLOCK"] = 2] = "BLOCK";
})(AUTH_TYPE || (AUTH_TYPE = {}));
function CASAuthentication(options) {
    if (!options || typeof options !== "object") {
        throw new Error("CAS Authentication was not given a valid configuration object.");
    }
    if (options.cas_url === undefined) {
        throw new Error("CAS Authentication requires a cas_url parameter.");
    }
    if (options.service_url === undefined) {
        throw new Error("CAS Authentication requires a service_url parameter.");
    }
    this.cas_version = "2.0";
    this._validateUri = "/serviceValidate";
    this._validate = function (body, callback) {
        xml2js_1.default.parseString(body, {
            trim: true,
            normalize: true,
            explicitArray: false,
            tagNameProcessors: [xml2js_1.default.processors.normalize, xml2js_1.default.processors.stripPrefix],
        }, function (err, result) {
            if (err) {
                return callback(new Error("Response from CAS server was bad."));
            }
            try {
                app_logs_1.logger.info(JSON.stringify(body));
                var failure = result.serviceresponse.authenticationfailure;
                if (failure) {
                    return callback(new Error("CAS authentication failed (" + failure.$.code + ")."));
                }
                var success = result.serviceresponse.authenticationsuccess;
                if (success) {
                    return callback(null, success.user, success.attributes);
                }
                else {
                    return callback(new Error("CAS authentication failed."));
                }
            }
            catch (err) {
                console.log(err);
                return callback(new Error("CAS authentication failed."));
            }
        });
    };
    this.cas_url = options.cas_url;
    var parsed_cas_url = url_1.default.parse(this.cas_url);
    this.request_client = parsed_cas_url.protocol === "http:" ? http_1.default : https_1.default;
    this.cas_host = parsed_cas_url.hostname;
    this.cas_port = parsed_cas_url.protocol === "http:" ? 80 : 443;
    this.cas_path = parsed_cas_url.pathname;
    this.service_url = options.service_url;
    this.renew = options.renew !== undefined ? !!options.renew : false;
    this.is_dev_mode = options.is_dev_mode !== undefined ? !!options.is_dev_mode : false;
    this.dev_mode_user = options.dev_mode_user !== undefined ? options.dev_mode_user : "";
    this.dev_mode_info = options.dev_mode_info !== undefined ? options.dev_mode_info : {};
    this.session_name = options.session_name !== undefined ? options.session_name : "cas_user";
    this.session_info = ["2.0", "3.0", "saml1.1"].indexOf(this.cas_version) >= 0 && options.session_info !== undefined ? options.session_info : false;
    this.destroy_session = options.destroy_session !== undefined ? !!options.destroy_session : false;
    this.bounce = this.bounce.bind(this);
    this.bounce_redirect = this.bounce_redirect.bind(this);
    this.block = this.block.bind(this);
    this.logout = this.logout.bind(this);
}
exports.CASAuthentication = CASAuthentication;
CASAuthentication.prototype.bounce = function (req, res, next) {
    this._handle(req, res, next, AUTH_TYPE.BOUNCE);
};
CASAuthentication.prototype.bounce_redirect = function (req, res, next) {
    this._handle(req, res, next, AUTH_TYPE.BOUNCE_REDIRECT);
};
CASAuthentication.prototype.block = function (req, res, next) {
    this._handle(req, res, next, AUTH_TYPE.BLOCK);
};
CASAuthentication.prototype._handle = function (req, res, next, authType) {
    if (req.session[this.session_name]) {
        if (authType === AUTH_TYPE.BOUNCE_REDIRECT) {
            res.redirect(req.session.cas_return_to);
        }
        else {
            next();
        }
    }
    else if (this.is_dev_mode) {
        req.session[this.session_name] = this.dev_mode_user;
        req.session[this.session_info] = this.dev_mode_info;
        next();
    }
    else if (authType === AUTH_TYPE.BLOCK) {
        res.sendStatus(401);
    }
    else if (req.query && req.query.ticket) {
        this._handleTicket(req, res, next);
    }
    else {
        this._login(req, res, next);
    }
};
CASAuthentication.prototype._login = function (req, res, next) {
    req.session.cas_return_to = req.query.returnTo || url_1.default.parse(req.url).path;
    var query = {
        service: this.service_url + url_1.default.parse(req.url).pathname,
        renew: this.renew,
    };
    res.setHeader("Access-Control-Allow-Origin", this.service_url);
    res.setHeader("Access-Control-Allow-Methods", "GET, POST, OPTIONS, PUT, PATCH, DELETE");
    res.setHeader("Access-Control-Allow-Headers", "X-Requested-With,content-type");
    res.setHeader("Access-Control-Allow-Credentials", true);
    res.status(409).json({
        url: this.cas_url +
            url_1.default.format({
                pathname: "/login",
                query: query,
            }),
    });
};
CASAuthentication.prototype.logout = function (req, res, next) {
    if (this.destroy_session) {
        req.session.destroy(function (err) {
            if (err) {
                console.log(err);
            }
        });
    }
    else {
        delete req.session[this.session_name];
        if (this.session_info) {
            delete req.session[this.session_info];
        }
    }
    res.redirect(this.cas_url + "/logout");
};
CASAuthentication.prototype._handleTicket = async function (req, res, next) {
    var requestOptions = {
        host: this.cas_host,
        port: this.cas_port,
        method: "GET",
        path: url_1.default.format({
            pathname: this.cas_path + this._validateUri,
            query: {
                service: this.service_url + url_1.default.parse(req.url).pathname,
                ticket: req.query.ticket,
            },
        }),
    };
    let requestPromise = new Promise((resolve, reject) => {
        var request = this.request_client.request(requestOptions, function (response) {
            response.setEncoding("utf8");
            var body = "";
            response.on("data", function (chunk) {
                return (body += chunk);
            }.bind(this));
            response.on("end", function () {
                this._validate(body, function (err, user, attributes) {
                    if (err) {
                        app_logs_1.logger.error(err);
                        res.sendStatus(401);
                        reject(err);
                    }
                    else {
                        req.session[this.session_name] = user;
                        if (this.session_info) {
                            req.session[this.session_info] = attributes || {};
                        }
                        next();
                    }
                    resolve(null);
                }.bind(this));
            }.bind(this));
            response.on("error", function (err) {
                app_logs_1.logger.error("Response error from CAS: ", err);
                res.sendStatus(401);
                reject(err);
            }.bind(this));
        }.bind(this));
        request.on("error", function (err) {
            app_logs_1.logger.error("Request error with CAS: ", err);
            res.sendStatus(401);
            reject(err);
        }.bind(this));
        request.end();
    });
    await requestPromise;
};
