"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const app_enums_1 = require("../../../../app.enums");
const dal_manager_1 = require("../../../../dal/dal.manager");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../api.user.v1.models");
const dal_access_psql_organization_1 = require("../../../../dal/access/psql/dal.access.psql.organization");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
module.exports.listUsersOfUserGroup = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userGroupId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.gid.value, "userGroupId");
    let request = new api_user_v1_models_1.RestUserApiV1.GenericSearchRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authResult = (0, api_securityhelper_1.authorizeForListUsersForUserGroup)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessUserGroup.listUsersOfUserGroup(organizationId, userGroupId, {
        take: request.take,
        skip: request.skip,
        filter: request.filter,
    });
    let result = {
        total: dbResult.pagination.total,
        skip: dbResult.pagination.skip,
        take: dbResult.pagination.take,
        items: dbResult.items.map((item) => {
            return {
                id: item.id,
                fullname: item.fullname,
                uniqueId: item.uniqueId,
                organizationUnits: item.organizationUnits,
                userGroups: item.userGroups,
                hasUserAccount: item.hasUserAccount,
                isDisabled: item.isDisabled,
            };
        }),
    };
    res.json(result);
});
module.exports.obtainUserSelectionSessionForUserGroup = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userGroupId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.gid.value, "userGroupId");
    let force = req.swagger.params.force.value;
    let authResult = (0, api_securityhelper_1.authorizeForObtainUserSelectionSessionForUserGroup)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUser.obtainUserSelectionSession(organizationId, {
            requestUserId: authResult.jwt.userId,
            type: app_enums_1.enums.UserSelectionSessionType.UserGroup,
            relatedItemId: userGroupId,
            forceToCreate: force,
            isAuthorizedForAllUsers: authResult.hasOrganizationWideWriteUserGroup && authResult.hasOrganizationWideReadBasic,
            authorizedOrganizationUnitIds: authResult.readBasicOrganizationUnitIds ?? null,
        }, trx);
    }, authResult.jwt.userId, organizationId);
    if (!dbResult.upserted) {
        throw (0, api_error_1.generateConflictError)({
            message: "There is already a session for that usergroup. Please see detailes.",
            details: dbResult.overwritten,
        });
    }
    let result = {
        sessionId: dbResult.sessionId,
        expirationUtc: dbResult.expirationUtc,
        justRefreshed: dbResult.justRefreshed,
        overwritten: dbResult.overwritten
            ? {
                ownerUserId: dbResult.overwritten.ownerUserId,
                sessionId: dbResult.overwritten.sessionId,
                expirationUtc: dbResult.overwritten.expirationUtc,
                captionLines: dbResult.overwritten.captionLines,
            }
            : undefined,
    };
    res.json(result);
});
module.exports.listUserGroups = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authorizationResult = (0, api_securityhelper_1.authorizeForListUserGroups)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.ListUserGroupsRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByFilter({ organizationId, apiFilter: request });
    res.json(dbResult);
});
module.exports.addUsergroup = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let request = new api_user_v1_models_1.RestUserApiV1.AddUserGroupRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let authorizationResult = (0, api_securityhelper_1.authorizeForAddUserGroup)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessUserGroup.addUserGroup(organizationId, request, authorizationResult.hasOrganizationWide, authorizationResult.jwt);
    res.json(dbResult);
});
module.exports.updateUserGroup = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const userGroupId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "userGroupId");
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const auth = (0, api_securityhelper_1.authorizeForUpdateUserGroup)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.UpdateUserGroupRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (!auth.hasOrganizationWide && !request.organizationUnitId) {
        throw (0, api_error_1.generateForbiddenError)({ message: "You do not have permission to add user group for organization wide!" });
    }
    const userGroup = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUserGroup.updateUserGroup({ organizationId, userGroupId, request, hasOrganizationWideRight: auth.hasOrganizationWide, jwt: auth.jwt, trx });
    }, auth.jwt.userId, organizationId);
    res.json(userGroup);
});
module.exports.removeUserGroup = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    new api_validatorhelper_1.ValidatorHelper().validateUUID("organizationId", false, req.swagger.params.oid.value).validateUUID("userGroupId", false, req.swagger.params.id.value).finalize();
    let organizationId = req.swagger.params.oid.value;
    let userGroupId = req.swagger.params.id.value;
    let authorizationResult = (0, api_securityhelper_1.authorizeForRemoveUserGroup)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const isUserFilterAccessRightActivated = await (0, dal_access_psql_organization_1.getOrganizationUserFilterAccessRightActivationStatus)(organizationId, trx);
        if (isUserFilterAccessRightActivated) {
            const userFilters = (await trx.query(`
						SELECT uf.*
						FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilter}" uf
						INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userFilterGroup}" as ufg
							ON uf.id = ufg."filterId"
						WHERE ufg."groupId" = $1
						`, [userGroupId])).rows;
            if (userFilters.length > 0) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "ERRORS.USER_FILTER.USER_FILTER_USAGE_GROUP", { userFilterNames: userFilters.map((elem) => elem.name).join(", ") }, true, true);
            }
        }
        await dal_manager_1.dbManager.accessUserGroup.removeUserGroup(organizationId, userGroupId, authorizationResult.hasOrganizationWide, authorizationResult.jwt, trx);
    }, authorizationResult.jwt.userId, organizationId);
    res.json({});
});
module.exports.listUserGroupsDetailed = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = new api_validatorhelper_1.ValidatorHelper().validateUUIDAndFinalize("organizationId", false, req.swagger.params.oid.value);
    let authorizationResult = (0, api_securityhelper_1.authorizeForListUserGroups)(req, organizationId);
    res.json(await dal_manager_1.dbManager.accessUserGroup.listUserGroups(organizationId, authorizationResult.hasOrganizationWide, authorizationResult.jwt));
});
module.exports.insertUserToGroup = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const groupId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "userGroupId");
    const { userIds } = new api_user_v1_models_1.RestUserApiV1.InsertUsersIntoGroup(req.body, new api_validatorhelper_1.ValidatorHelper());
    const auth = (0, api_securityhelper_1.authorizeForAddUserGroup)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUserGroup.insertUsersIntoGroup(organizationId, auth.jwt.userId, trx, { groupId, userIds });
    }, auth.jwt.userId, organizationId);
    res.json({});
});
module.exports.removeUserFromGroup = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const groupId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "userGroupId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForAddUserGroup)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUserGroup.removeUserFromGroup(organizationId, auth.jwt.userId, trx, { groupId, userId });
    }, auth.jwt.userId, organizationId);
    res.json({});
});
