"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.authorizeWriteForUserFilter = void 0;
const dal_manager_1 = require("../../../../dal/dal.manager");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_user_filter_models_1 = require("../models/api.user.v1.user-filter.models");
const restapi_1 = require("../../../../lib/es/models/restapi");
const api_securityhelper_1 = require("../../../api.securityhelper");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const app_auth_1 = require("../../../../app.auth");
const dal_access_psql_common_1 = require("../../../../dal/access/psql/dal.access.psql.common");
const dal_constants_1 = require("../../../../dal/dal.constants");
const app_enums_1 = require("../../../../app.enums");
const api_error_1 = require("../../../api.error");
async function checkAuthForUserFilter(organizationId, userId, filterResult, checkedAuths, trx) {
    let authorizedUsersQueryAndBindings = (0, dal_access_psql_common_1.getReportUserFilterForPgClient)({
        organizationId: organizationId,
        requesterUserId: userId,
        idBasedUserFilter: {
            userOrganizationStatus: dal_constants_1.DalConstants.IdentityStatusType.Enabled,
        },
        requiredOrganizationUnitWidePermissions: checkedAuths,
        bindingKeys: [],
        specificSelectItems: ["userId"],
    });
    let authorizedUsers = (await trx.query(authorizedUsersQueryAndBindings.query, authorizedUsersQueryAndBindings.bindingKeys)).rows.map((elem) => elem.userId);
    if (!filterResult.every((elem) => authorizedUsers.includes(elem))) {
        (0, api_securityhelper_1.throwAuthError)(null, checkedAuths, checkedAuths);
    }
}
function authForUserFilterList(req, organizationId, userFilterUsageType) {
    const usageTypes = [];
    if (req.auth_token && req.auth_token instanceof app_auth_1.UserJwtPayload) {
        const jwt = req.auth_token;
        if (jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.identity.getAccessRight())) {
            usageTypes.push(restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS);
        }
        if (jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.ppermission.getRead())) {
            usageTypes.push(restapi_1.UserFilterUsageType.PERMISSIONS);
        }
        if (jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.scheduled_job.getRead())) {
            usageTypes.push(restapi_1.UserFilterUsageType.NOTIFICATION);
        }
        if (jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.attendancePlan.getRead())) {
            usageTypes.push(restapi_1.UserFilterUsageType.WORK_PLAN_TEMPLATE);
        }
        if (usageTypes.length === 0 || (userFilterUsageType?.length > 0 && !userFilterUsageType.every((elem) => usageTypes.includes(elem)))) {
            const missingPermissions = jwt.getMissingPermissions(organizationId, [predefined_permissions_1.Permissions.ppermission.getRead(), predefined_permissions_1.Permissions.scheduled_job.getRead(), predefined_permissions_1.Permissions.attendancePlan.getRead()], true);
            (0, api_securityhelper_1.throwAuthError)(null, missingPermissions, [predefined_permissions_1.Permissions.ppermission.getRead(), predefined_permissions_1.Permissions.scheduled_job.getRead(), predefined_permissions_1.Permissions.attendancePlan.getRead()]);
        }
    }
    return usageTypes;
}
async function authorizeWriteForUserFilter(req, organizationId, filterId, trx) {
    const usageType = await dal_manager_1.dbManager.accessUserFilter.getFilterUsageType(organizationId, filterId, trx);
    if (req.auth_token && req.auth_token instanceof app_auth_1.UserJwtPayload) {
        const jwt = req.auth_token;
        switch (usageType) {
            case restapi_1.UserFilterUsageType.PERMISSIONS: {
                if (!jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.ppermission.getWrite())) {
                    (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.ppermission.getWrite()], [predefined_permissions_1.Permissions.ppermission.getWrite()]);
                }
                break;
            }
            case restapi_1.UserFilterUsageType.NOTIFICATION: {
                if (!jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.scheduled_job.getWrite())) {
                    (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.scheduled_job.getWrite()], [predefined_permissions_1.Permissions.scheduled_job.getWrite()]);
                }
                break;
            }
            case restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS: {
                if (!jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.identity.getAccessRight())) {
                    (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.identity.getAccessRight()], [predefined_permissions_1.Permissions.identity.getAccessRight()]);
                }
                break;
            }
            case restapi_1.UserFilterUsageType.WORK_PLAN_TEMPLATE: {
                if (!jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.attendancePlan.getWrite())) {
                    (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.attendancePlan.getWrite()], [predefined_permissions_1.Permissions.attendancePlan.getWrite()]);
                }
                break;
            }
        }
    }
    return usageType;
}
exports.authorizeWriteForUserFilter = authorizeWriteForUserFilter;
async function authorizeForGetUserFilterDetail(req, organizationId, filterId, trx) {
    const usageType = await dal_manager_1.dbManager.accessUserFilter.getFilterUsageType(organizationId, filterId, trx);
    if (req.auth_token && req.auth_token instanceof app_auth_1.UserJwtPayload) {
        const jwt = req.auth_token;
        switch (usageType) {
            case restapi_1.UserFilterUsageType.PERMISSIONS: {
                if (!jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.ppermission.getRead())) {
                    (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.ppermission.getRead()], [predefined_permissions_1.Permissions.ppermission.getRead()]);
                }
                break;
            }
            case restapi_1.UserFilterUsageType.NOTIFICATION: {
                if (!jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.scheduled_job.getRead())) {
                    (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.scheduled_job.getRead()], [predefined_permissions_1.Permissions.scheduled_job.getRead()]);
                }
                break;
            }
            case restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS: {
                if (!jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.identity.getAccessRight())) {
                    (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.identity.getAccessRight()], [predefined_permissions_1.Permissions.identity.getAccessRight()]);
                }
                break;
            }
            case restapi_1.UserFilterUsageType.WORK_PLAN_TEMPLATE: {
                if (!jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.attendancePlan.getRead())) {
                    (0, api_securityhelper_1.throwAuthError)(null, [predefined_permissions_1.Permissions.attendancePlan.getRead()], [predefined_permissions_1.Permissions.attendancePlan.getRead()]);
                }
                break;
            }
        }
    }
}
module.exports.listUserFilter = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = req.auth_token.userId;
    const { value, error } = api_user_v1_user_filter_models_1.ListUserFilterRequestSchema.validate(req.body);
    const accessibleUserFilterTypes = authForUserFilterList(req, organizationId, value.usageType);
    value.usageType = value.usageType ? value.usageType : accessibleUserFilterTypes;
    if (error) {
        res.status(400).json({ message: error.stack });
    }
    const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
        return await dal_manager_1.dbManager.accessUserFilter.listUserFilters(organizationId, value, userId, client);
    }, userId, organizationId);
    res.json(result);
});
module.exports.deleteUserFilter = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    try {
        let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
        const userId = (0, api_validatorhelper_1.validateUUID)(req.auth_token.userId, "userId");
        const filterId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.ufid.value, "userFilterId");
        const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
            const userFilterUsageType = await authorizeWriteForUserFilter(req, organizationId, filterId, client);
            if (userFilterUsageType === restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS) {
                const userRights = await dal_manager_1.dbManager.accessUserFilter.getUserRightsOverUserFilters(organizationId, [filterId], userId, client);
                if (userRights.length === 0 || userRights[0]?.grant) {
                    return await dal_manager_1.dbManager.accessUserFilter.deleteUserFilter(organizationId, filterId, client);
                }
                else {
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.NOGRANTONACP", null, true);
                }
            }
            else {
                return await dal_manager_1.dbManager.accessUserFilter.deleteUserFilter(organizationId, filterId, client);
            }
        }, userId, organizationId);
        res.json(result);
    }
    catch (error) {
        if (error.code === app_enums_1.enums.PostgresErrorCodes.INVALID_ACCESS_CONTROL_POINT_RIGHTS) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INVALID_ACCESS_CONTROL_POINT_RIGHTS, "ERRORS.GENERAL.LASTUSERWITHCONFIG", null, true);
        }
        else if (error.statusCode === app_enums_1.enums.HttpStatusCode.CONFLICT) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.NOGRANTONACP", null, true);
        }
        else {
            res.status(400).json({ body: error });
        }
    }
});
module.exports.getUserFilterDetail = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.auth_token.userId, "userId");
    const filterId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.ufid.value, "userFilterId");
    const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
        await authorizeForGetUserFilterDetail(req, organizationId, filterId, client);
        return await dal_manager_1.dbManager.accessUserFilter.getUserFilterDetail(organizationId, filterId, client);
    }, userId, organizationId);
    res.json(result);
});
module.exports.upsertUserFilter = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    try {
        let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
        const userId = req.auth_token.userId;
        const { value, error } = api_user_v1_user_filter_models_1.UserFilterV2Schema.validate(req.body);
        if (error) {
            res.status(400).json({ name: "API Bad Request Error", message: "Request validation error", body: error });
        }
        else {
            const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
                const usageType = await authorizeWriteForUserFilter(req, organizationId, req.body.id, client);
                if (req.body.id) {
                    if (usageType === restapi_1.UserFilterUsageType.USER_ACCESS_RIGHTS) {
                        const userRights = await dal_manager_1.dbManager.accessUserFilter.getUserRightsOverUserFilters(organizationId, [value.id], userId, client);
                        if (userRights.length === 0 || userRights[0]?.grant) {
                            return await dal_manager_1.dbManager.accessUserFilter.updateUserFilter(organizationId, value, client);
                        }
                        else {
                            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.NOGRANTONACP", null, true);
                        }
                    }
                    else {
                        return await dal_manager_1.dbManager.accessUserFilter.updateUserFilter(organizationId, value, client);
                    }
                }
                else {
                    return await dal_manager_1.dbManager.accessUserFilter.insertUserFilter(organizationId, value, client);
                }
            }, userId, organizationId);
            res.json(result);
        }
    }
    catch (error) {
        if (error.code === app_enums_1.enums.PostgresErrorCodes.INVALID_ACCESS_CONTROL_POINT_RIGHTS) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INVALID_ACCESS_CONTROL_POINT_RIGHTS, "ERRORS.GENERAL.LASTUSERWITHCONFIG", null, true);
        }
        else if (error.statusCode === app_enums_1.enums.HttpStatusCode.CONFLICT) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.NOGRANTONACP", null, true);
        }
        else {
            res.status(400).json({ body: error });
        }
    }
});
module.exports.searchUserFilter = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = req.auth_token.userId;
    const { value, error } = api_user_v1_user_filter_models_1.SearchUserFilterRequestSchema.validate(req.body);
    const accessibleUserFilterTypes = authForUserFilterList(req, organizationId, value.usageType ? [value.usageType] : []);
    if (error) {
        res.status(400).json({ message: error.stack });
    }
    const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
        return await dal_manager_1.dbManager.accessUserFilter.searchUserFilterByName(organizationId, Object.assign(value, { authorizedUsageTypes: accessibleUserFilterTypes }), client);
    }, userId, organizationId);
    res.json(result);
});
module.exports.listUserFilterByIds = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = req.auth_token.userId;
    const { value, error } = api_user_v1_user_filter_models_1.ListUserFilterByIdsRequestSchema.validate(req.body);
    const accessibleUserFilterTypes = authForUserFilterList(req, organizationId);
    if (error) {
        res.status(400).json({ message: error.stack });
    }
    const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
        return await dal_manager_1.dbManager.accessUserFilter.listUserFilterByIds(organizationId, Object.assign(value, { authorizedUsageTypes: accessibleUserFilterTypes }), userId, client);
    }, userId, organizationId);
    res.json(result);
});
module.exports.userFilterResult = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = req.auth_token.userId;
    const { value, error } = api_user_v1_user_filter_models_1.UserFilterResultRequestSchema.validate(req.body);
    if (error) {
        res.status(400).json({ message: error.stack });
    }
    const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
        await authorizeForGetUserFilterDetail(req, organizationId, value.userFilter.id, client);
        return await dal_manager_1.dbManager.accessUserFilter.getFilterResultByDefinition(organizationId, value, client);
    }, userId, organizationId);
    res.json(result);
});
module.exports.getUserFilterAccessControlPointMappings = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.auth_token.userId, "userId");
    const filterId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.ufid.value, "userFilterId");
    const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
        await authorizeForGetUserFilterDetail(req, organizationId, filterId, client);
        return await dal_manager_1.dbManager.accessUserFilter.getUserFilterAccessControlPointMappings(organizationId, filterId, client);
    }, userId, organizationId);
    res.json(result);
});
module.exports.updateUserFilterAccessControlPointMappings = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    try {
        let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
        const userId = (0, api_validatorhelper_1.validateUUID)(req.auth_token.userId, "userId");
        const filterId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.ufid.value, "userFilterId");
        const { value, error } = api_user_v1_user_filter_models_1.UpdateUserFilterAcpMappingsRequestSchema.validate(req.body);
        if (error) {
            res.status(400).json({ name: "API Bad Request Error", message: error.stack, statusCode: 400, body: error });
        }
        await dal_manager_1.dbManager.organizationTransaction(async (client) => {
            await authorizeWriteForUserFilter(req, organizationId, req.body.id, client);
            return await dal_manager_1.dbManager.accessUserFilter.updateUserFilterAccessControlPointMappings(organizationId, {
                filterId,
                accessControlPoints: value.items,
            }, client);
        }, userId, organizationId);
        res.json();
    }
    catch (error) {
        if (error.code === app_enums_1.enums.PostgresErrorCodes.INVALID_ACCESS_CONTROL_POINT_RIGHTS) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INVALID_ACCESS_CONTROL_POINT_RIGHTS, "ERRORS.GENERAL.LASTUSERWITHCONFIG", null, true);
        }
        else {
            res.status(400).json({ body: error });
        }
    }
});
module.exports.getAccessControlPointUserFilterMappings = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.auth_token.userId, "userId");
    const acpId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "acpId");
    const result = await dal_manager_1.dbManager.organizationTransaction(async (client) => {
        return await dal_manager_1.dbManager.accessUserFilter.getAccessControlPointUserFilterMappings(organizationId, acpId, client);
    }, userId, organizationId);
    res.json(result);
});
module.exports.updateAccessControlPoinUserFiltertMappings = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    try {
        let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
        const userId = (0, api_validatorhelper_1.validateUUID)(req.auth_token.userId, "userId");
        const acpId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "acpId");
        const { value, error } = api_user_v1_user_filter_models_1.UpdateAcpUserFilterMappingsRequestSchema.validate(req.body);
        if (error) {
            res.status(400).json({ message: error.stack });
        }
        await dal_manager_1.dbManager.organizationTransaction(async (client) => {
            await authorizeWriteForUserFilter(req, organizationId, req.body.id, client);
            return await dal_manager_1.dbManager.accessUserFilter.updateAccessControlPoinUserFiltertMappings(organizationId, {
                acpId,
                filters: value.items,
            }, client);
        }, userId, organizationId);
        res.json();
    }
    catch (error) {
        if (error.code === app_enums_1.enums.PostgresErrorCodes.INVALID_ACCESS_CONTROL_POINT_RIGHTS) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INVALID_ACCESS_CONTROL_POINT_RIGHTS, "ERRORS.GENERAL.LASTUSERWITHCONFIG", null, true);
        }
        else {
            res.status(400).json({ body: error });
        }
    }
});
