"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const app_enums_1 = require("../../../../app.enums");
const dal_constants_1 = require("../../../../dal/dal.constants");
const dal_manager_1 = require("../../../../dal/dal.manager");
const restapi_1 = require("../../../../lib/es/models/restapi");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../api.user.v1.models");
const business_hooks_1 = require("../../../../business/business.hooks");
const i18n_1 = __importDefault(require("i18n"));
module.exports.getUserDefinedList = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForGetIdentityBasic)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.OrganizationUserDefinedListRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessOrganization.getOrganizationUserDefinedList({ ...request, organizationId, trx });
    }, authResult.jwt.userId, organizationId);
    res.json(dbResult);
});
module.exports.upsertUserDefinedListValue = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userDefinedListId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "userDefinedListId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.OrganizationUpsertUserDefinedListValueRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessOrganization.upsertOrganizationUserDefinedListValue(organizationId, userDefinedListId, request.value, request.relatedValueId, request.id);
    res.json(dbResult);
});
module.exports.deleteUserDefinedListValue = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userDefinedListId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "userDefinedListId");
    const userDefinedListValueId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.vid.value, "userDefinedListValueId");
    const authResult = (0, api_securityhelper_1.authorizeForGetIdentityBasic)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.OrganizationUserDefinedListRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.systemTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessOrganization.deleteOrganizationUserDefinedListValue({ organizationId, userDefinedListId, userDefinedListValueId, trx });
    });
    res.json();
});
module.exports.uploadOrganizationLogoFile = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    const file = req.swagger.params.upfile.value;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessOrganization.setOrganizationSettings({ organizationId, settings: { logo: file.buffer }, trx });
    }, authResult.userId, organizationId);
    res.json();
});
module.exports.getOrganizationLogoFile = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const organizationSettings = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessOrganization.getOrganizationSettings({ organizationId, trx });
    });
    const logo = organizationSettings.logo ? Buffer.from(organizationSettings.logo, "base64") : null;
    if (logo) {
        res.writeHead(200, {
            "Content-Type": "image/svg+xml",
            "Content-Length": logo.length,
        });
    }
    else {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "logo not found" });
    }
    res.end(logo);
});
module.exports.removeOrganizationLogoFile = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessOrganization.removeOrganizationLogoFile(organizationId, trx);
    }, authResult.userId, organizationId);
    res.json();
});
module.exports.getOrganizationSettings = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForListAvailableRoles)(req, organizationId);
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const orgModules = await dal_manager_1.dbManager.accessOrganization.listOrganizationModules(organizationId, trx);
        let organizationSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationSettings({ organizationId, trx });
        const customNotificationSettings = [];
        if (organizationSettings.notification.mediumSettings.custom) {
            for (const key of Object.keys(organizationSettings.notification.mediumSettings.custom)) {
                if (parseInt(key) !== restapi_1.NotificationType.SendingVerificationToken) {
                    customNotificationSettings.push({
                        type: parseInt(key),
                        mediums: organizationSettings.notification.mediumSettings.custom[key],
                    });
                }
            }
        }
        let availableTypes = dal_constants_1.DalConstants.organizationBaseModuleNotificationTypes;
        if (orgModules.find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.PACS)?.enabled) {
            availableTypes = availableTypes.concat(...dal_constants_1.DalConstants.organizationPACSModuleNotificationTypes);
        }
        if (orgModules.find((m) => m.module === app_enums_1.enums.ArmonApplicationModule.Visitor)?.enabled) {
            availableTypes = availableTypes.concat(...dal_constants_1.DalConstants.organizationVisitorModuleNotificationTypes);
        }
        if (!organizationSettings.notification.availableMediums.sms) {
            availableTypes = availableTypes.filter((type) => type !== restapi_1.NotificationType.SMSBalance);
        }
        if (organizationSettings.notification.smtpSettings?.enabled) {
            delete organizationSettings.notification.smtpSettings.config.tls;
        }
        else {
            delete organizationSettings.notification.smtpSettings;
        }
        return {
            locale: organizationSettings.locale,
            helpUrl: organizationSettings.helpUrl,
            notification: {
                enabledTypes: organizationSettings.notification.enabledTypes,
                availableTypes: availableTypes,
                smsBalance: organizationSettings.notification.smsBalance,
                availableMediums: organizationSettings.notification.availableMediums,
                smtpSettings: organizationSettings.notification.smtpSettings,
                smsSettings: organizationSettings.notification.smsSettings,
                mediumSettings: {
                    general: organizationSettings.notification.mediumSettings.general,
                    custom: customNotificationSettings,
                },
            },
            webRtc: organizationSettings.webRtc,
            checkMobileDeviceMismatchOnLogin: organizationSettings.checkMobileDeviceMismatchOnLogin,
            multiFactorAuthenticationSettings: organizationSettings.multiFactorAuthenticationSettings,
        };
    }, authResult.userId, organizationId);
    const hook = business_hooks_1.armonHookManager.getAfterGetOrganizationSettings(organizationId);
    if (hook) {
        await hook(result);
    }
    res.json(result);
});
module.exports.getOrganizationSettingsIntegrations = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const jwt = req.auth_token;
    const userId = jwt.userId;
    let result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return await dal_manager_1.dbManager.accessOrganization.getOrganizationSettingsIntegrations({ organizationId, trx });
    }, userId, organizationId);
    if (!result) {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "integrations not found" });
    }
    const filteredResult = result.map(({ type, enabled, config: { secretId, ...config } }) => ({
        type,
        enabled,
        config,
    }));
    res.json(filteredResult);
});
module.exports.setOrganizationSettings = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForOrganizationSettings)(req, organizationId);
    const settings = req.body;
    const hook = business_hooks_1.armonHookManager.getBeforeUpdateOrganizationSettings(organizationId);
    if (hook) {
        const result = await hook(settings);
        if (!result.isValid) {
            let message = i18n_1.default.__({ phrase: result.errorMessage, locale: req.locale ?? "tr" }, result.replacements);
            res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({ message: message, noReport: true, showAsModal: true });
            return;
        }
    }
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const organizationSettings = await dal_manager_1.dbManager.accessOrganization.getOrganizationSettings({ organizationId, trx });
        if (settings.multiFactorAuthenticationSettings) {
            const defaultNotificationMedium = { web: false, email: false, pushNotification: false, sms: false };
            if (settings.multiFactorAuthenticationSettings.method) {
                switch (settings.multiFactorAuthenticationSettings.method) {
                    case app_enums_1.enums.MultifactorAuthenticationVerificationMethod.SMS: {
                        defaultNotificationMedium.sms = true;
                        break;
                    }
                    case app_enums_1.enums.MultifactorAuthenticationVerificationMethod.EMAIL: {
                        defaultNotificationMedium.email = true;
                        break;
                    }
                }
                if (!organizationSettings.notification.mediumSettings.custom) {
                    organizationSettings.notification.mediumSettings.custom = {
                        [restapi_1.NotificationType.SendingVerificationToken]: defaultNotificationMedium,
                    };
                }
                else {
                    organizationSettings.notification.mediumSettings.custom[restapi_1.NotificationType.SendingVerificationToken] = defaultNotificationMedium;
                }
                settings.notification = { mediumSettings: { custom: [] } };
                Object.keys(organizationSettings.notification.mediumSettings.custom ?? {}).forEach((elem) => settings.notification.mediumSettings.custom.push({
                    type: parseInt(elem),
                    mediums: organizationSettings.notification.mediumSettings.custom[elem],
                }));
            }
            switch (settings.multiFactorAuthenticationSettings.option) {
                case app_enums_1.enums.MultifactorAuthenticationOption.DISABLE:
                    settings.notification = { mediumSettings: { custom: [] } };
                    Object.keys(organizationSettings.notification.mediumSettings.custom ?? {}).forEach((elem) => {
                        if (parseInt(elem) !== restapi_1.NotificationType.SendingVerificationToken) {
                            settings.notification.mediumSettings.custom.push({
                                type: parseInt(elem),
                                mediums: organizationSettings.notification.mediumSettings.custom[elem],
                            });
                        }
                    });
                    await dal_manager_1.dbManager.accessUser.disableAllUsersMultFactorAuthSettings(organizationId, trx);
                    break;
                case app_enums_1.enums.MultifactorAuthenticationOption.OPTIONAL:
                    await dal_manager_1.dbManager.accessUser.disableAllUsersMultFactorAuthSettings(organizationId, trx);
                    break;
                case app_enums_1.enums.MultifactorAuthenticationOption.REQUIRED:
                    await dal_manager_1.dbManager.accessUser.enableAllUsersMultiFactorAuthSettings(organizationId, {
                        enabledStatus: true,
                    }, {
                        [restapi_1.NotificationType.SendingVerificationToken]: defaultNotificationMedium,
                    }, trx);
                    break;
            }
        }
        await dal_manager_1.dbManager.accessOrganization.setOrganizationSettings({ organizationId, settings, trx });
    }, authResult.userId, organizationId);
    res.json();
});
module.exports.userExpression = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSystemWrite)(req, organizationId);
    let request = req.body;
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessOrganization.setOrganizationSettings({ organizationId, settings: { userExpression: request.userExpressionType }, trx });
    }, authResult.userId, organizationId);
    res.json();
});
