"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const lodash_1 = __importDefault(require("lodash"));
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_filesystem_1 = require("../../../../app.filesystem");
const app_httpserver_1 = require("../../../../app.httpserver");
const app_logs_1 = require("../../../../app.logs");
const business_hooks_1 = require("../../../../business/business.hooks");
const business_main_1 = require("../../../../business/business.main");
const business_notification_1 = require("../../../../business/business.notification");
const business_roles_1 = require("../../../../business/business.roles");
const report_util_1 = require("../../../../business/report/report.util");
const dal_memcache_1 = require("../../../../dal/access/dal.memcache");
const dal_constants_1 = require("../../../../dal/dal.constants");
const dal_manager_1 = require("../../../../dal/dal.manager");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const terminal_1 = require("../../../../lib/es/models/terminal");
const utils_1 = require("../../../../lib/v8/utils");
const messageBroker_server_to_report_pub_1 = require("../../../../messageBroker/messageBroker.server-to-report.pub");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../api.user.v1.models");
const api_arvento_1 = require("../../../api.arvento");
const api_util_2 = require("../../../api.util");
const restapi_1 = require("../../../../lib/es/models/restapi");
module.exports.deleteUserOrganizationProfilePhoto = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    let userID = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let photoId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.pid.value, "photoId");
    let result = await dal_manager_1.dbManager.accessIdentity.deleteUserOrganizationProfilePhoto(organizationId, userID, photoId);
    if (result) {
        await app_filesystem_1.fileSystem.deleteFile(organizationId, photoId);
    }
    else {
        next((0, api_error_1.generateNotFoundApiError)({ message: "Could not delete file from file system" }));
        return;
    }
    res.json();
});
module.exports.listUserOrganizationProfilePhotos = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let result = await dal_manager_1.dbManager.accessIdentity.listUserOrganizationProfilePhotos({ organizationId, userId });
    res.json(result);
});
module.exports.addUserOrganizationProfilePhoto = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let type = req.swagger.params.type.value;
    let fileData = req.swagger.params.upfile.value;
    let result = await dal_manager_1.dbManager.accessIdentity.addUserOrganizationProfilePhoto({ organizationId, userId, type });
    let fileSystemResult = await app_filesystem_1.fileSystem.addFile(organizationId, result, fileData);
    res.json({
        PhotoId: result,
    });
});
module.exports.getUserOrganizationProfilePhoto = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForGetIdentityBasic)(req, organizationId);
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let photoId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.pid.value, "photoId");
    let file = await app_filesystem_1.fileSystem.getFileData(organizationId, photoId, "base64");
    res.json({
        photoId: photoId,
        data: file,
    });
});
module.exports.getUserOrganizationProfileThumbnail = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForGetIdentityBasic)(req, organizationId);
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let dbResult = await dal_manager_1.dbManager.accessIdentity.geUserOrganizationThumbnail({
        organizationId,
        userId,
    });
    if (dbResult === null || dbResult === undefined) {
        return res.json({
            thumbnail: null,
        });
    }
    else {
        res.json(dbResult);
    }
});
module.exports.getSelfUserOrganizationProfileThumbnail = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let dbResult = await dal_manager_1.dbManager.accessIdentity.geUserOrganizationThumbnail({
        organizationId: organizationId,
        userId: jwt.userId,
    });
    if (dbResult === null || dbResult === undefined) {
        res.json({
            thumbnail: null,
        });
    }
    else {
        res.json(dbResult);
    }
});
module.exports.upsertUserOrganizationProfileThumbnail = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let thumbnail = req.body.data ? req.body.data : null;
    if (thumbnail) {
        const validatorHelper = new api_validatorhelper_1.ValidatorHelper();
        (0, api_validatorhelper_1.validateThumbnail)(thumbnail, validatorHelper, true);
    }
    else {
        throw (0, api_error_1.generateBadRequestApiError)({
            message: "Thumbnail data is required",
        });
    }
    let dbResult = await dal_manager_1.dbManager.accessIdentity.upsertUserOrganizationThumbnail({
        organizationId,
        userId,
        thumbnail,
    });
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "Could not load thumbnail" }));
        return;
    }
    res.json({});
});
module.exports.searchIdentityForExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let jwt = req.auth_token;
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = req.body;
    let dbRequest = {
        organizationId: organizationId,
        take: app_config_1.appConfig.reportExportRowLimit + 1,
        skip: 0,
        status: request.status,
        profileFilter: request.profileFilter,
        organizationUnits: request.organizationUnits || [],
        userGroupIds: request.userGroupIds,
        workplans: request.workplans || [],
    };
    if (!authResult.hasOrganizationWide) {
        dbRequest.organizationUnits.push(...authResult.jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.identity.getReadBasic()).map((unitId) => {
            return {
                id: unitId,
                includeDescendants: true,
            };
        }));
    }
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = dbRequest;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.Identity,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.searchIdentityForPdf = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let jwt = req.auth_token;
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = req.body;
    let dbRequest = {
        organizationId: organizationId,
        take: app_config_1.appConfig.reportExportRowLimit + 1,
        skip: 0,
        status: request.status,
        profileFilter: request.profileFilter,
        organizationUnits: request.organizationUnits || [],
        userGroupIds: request.userGroupIds,
    };
    if (!authResult.hasOrganizationWide) {
        dbRequest.organizationUnits.push(...authResult.jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.identity.getReadBasic()).map((unitId) => {
            return {
                id: unitId,
                includeDescendants: true,
            };
        }));
    }
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = dbRequest;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.Identity,
                format: app_enums_1.enums.ReportFormat.Pdf,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.searchIdentity = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = req.body;
    let dbRequest = {
        organizationId: organizationId,
        take: request.take,
        skip: request.skip,
        status: request.status,
        profileFilter: request.profileFilter,
        organizationUnits: request.organizationUnits || [],
        userGroupIds: request.userGroupIds,
        credentialExtensionFilters: request.credentialExtensionFilters || [],
        hasMail: request.hasMail || false,
        permissionsOr: request.permissionsOr,
        permissions: request.permissions,
        workplans: request.workplans || [],
    };
    if (!authResult.hasOrganizationWide && dbRequest.organizationUnits.length === 0) {
        if (dbRequest.status !== app_enums_1.enums.IdentityStatusType.Disabled || !authResult.jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.system.getOrganizationWidePassiveUsersView())) {
            dbRequest.organizationUnits.push(...authResult.jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.identity.getReadBasic()).map((unitId) => {
                return {
                    id: unitId,
                    includeDescendants: true,
                };
            }));
        }
    }
    let dbResult = await dal_manager_1.dbManager.accessIdentity.searchIdentity(dbRequest);
    let result = {
        total: dbResult.pagination.total,
        skip: dbResult.pagination.skip,
        take: dbResult.pagination.take,
        items: dbResult.items.map((item) => {
            return {
                id: item.id,
                fullname: item.fullname,
                uniqueId: item.uniqueId,
                isDisabled: item.isDisabled,
                isAnonymized: item.isAnonymized,
                hasUserAccount: item.hasUserAccount,
                organizationUnits: item.organizationUnits,
                userGroups: item.userGroups,
                hasForbiddance: item.hasForbiddance,
                userCaptions: item.userCaptions,
            };
        }),
    };
    res.json(result);
});
module.exports.searchIdentityExactForExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = req.body;
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.Identity,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.searchIdentityExactForPdf = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let jwt = req.auth_token;
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = req.body;
    try {
        let reportResult = await new Promise((resolve, reject) => {
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.Identity,
                format: app_enums_1.enums.ReportFormat.Pdf,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.searchIdentityExact = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.SearchIdentityExactRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessIdentity.searchIdentityExact(organizationId, request);
    let result = {
        total: dbResult.pagination.total,
        skip: dbResult.pagination.skip,
        take: dbResult.pagination.take,
        items: dbResult.items.map((item) => {
            return {
                id: item.id,
                fullname: item.fullname,
                uniqueId: item.uniqueId,
                isDisabled: item.isDisabled,
                isAnonymized: item.isAnonymized,
                hasUserAccount: item.hasUserAccount,
                organizationUnits: item.organizationUnits,
                userGroups: item.userGroups,
                hasForbiddance: item.hasForbiddance,
                userCaptions: item.userCaptions,
            };
        }),
    };
    res.json(result);
});
module.exports.listIdentity = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = req.body;
    let dbResult = await dal_manager_1.dbManager.accessIdentity.listIdentity(organizationId, request.userIds, request.take, request.skip);
    res.json({
        total: dbResult.pagination.total,
        skip: dbResult.pagination.skip,
        take: dbResult.pagination.take,
        items: dbResult.items
            ? dbResult.items.map((item) => {
                return {
                    id: item.id,
                    fullname: item.fullname,
                    uniqueId: item.uniqueId,
                    isDisabled: item.isDisabled,
                    hasUserAccount: item.hasUserAccount,
                    organizationUnits: item.organizationUnits,
                    userGroups: item.userGroups,
                };
            })
            : [],
    });
});
module.exports.findMemberByCredential = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let visitorRead = (0, api_securityhelper_1.hasVisitorRead)(req, organizationId);
    let permittedUnitIds = authResult.hasOrganizationWide ? null : authResult.jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.identity.getReadBasic());
    let request = req.body;
    let dbResult = await dal_manager_1.dbManager.accessIdentity.findMemberByCredential(organizationId, visitorRead, request.credentialData, request.credentialType, permittedUnitIds);
    let result = {
        existance: dbResult.existance,
        credentialType: dbResult.credentialType,
        visitorInfo: dbResult.visitorInfo,
        roleId: dbResult ? dbResult.roleId : "",
        member: dbResult && dbResult.member
            ? {
                id: dbResult.member.id,
                fullname: dbResult.member.fullname,
                uniqueId: dbResult.member.uniqueId,
                organizationUnits: dbResult.member.organizationUnits,
                userGroups: dbResult.member.userGroups,
            }
            : null,
    };
    res.json(result);
});
module.exports.getSelfDeputyInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForGetIdentityBasic)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessIdentity.getSelfDeputyInfo(organizationId, authResult.jwt.userId);
    res.json({
        items: dbResult
            ? dbResult.items.map((item) => {
                return {
                    organizationUnitName: item.organizationUnitName,
                    principalFullName: item.principalFullName,
                    startDateTime: item.startDateTime,
                    endDateTime: item.endDateTime,
                };
            })
            : [],
    });
});
module.exports.insertIdentity = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const request = new api_user_v1_models_1.RestUserApiV1.InsertIdentityRequest().validate(req.body, new api_validatorhelper_1.ValidatorHelper());
    if ((0, api_util_2.hasDuplicateUuids)(request.organizationUnits.map((unit) => unit.organizationUnitId))) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.DUPLICATEORGANIZATIONUNITS");
    }
    if ((0, api_util_2.hasDuplicateUuids)(request.userGroupIds)) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.DUPLICATEUSERGROUPS");
    }
    const authResult = (0, api_securityhelper_1.authorizeForInsertIdentity)(req, organizationId);
    const { r: requesterRoleId } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    if (!(await (0, business_roles_1.canUserChangeRole)(organizationId, requesterRoleId, request.organizationProfile.roleId))) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.INVALIDROLEDEF");
    }
    if (request.accessRights && request.accessRights.length > 0) {
        if (!authResult.hasPermissionForOrganizationOrAnyUnit(organizationId, predefined_permissions_1.Permissions.system.getRead())) {
            const userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.userId);
            if (!userAccessRights.some((u) => request.accessRights.some((r) => r.accessControlPointId == u.accessControlPointId) && u.grant)) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.UNABLETOCHANGEACCESSRIGHT");
            }
        }
    }
    if (request.organizationUnits?.length > 0 && request.addDefaultAccessRightsOfUnits) {
        let addedOrganizationUnitDefaultAccessControlPointRights = null;
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            addedOrganizationUnitDefaultAccessControlPointRights = await dal_manager_1.dbManager.accessOrganizationUnit.getDefaultAccessControlPointRightsOfOrganizationUnits(organizationId, request.organizationUnits?.map((m) => m.organizationUnitId) ?? [], trx);
        }, authResult.userId, organizationId);
        if (!request.accessRights) {
            request.accessRights = [];
        }
        for (const unitDefaultAccessControlPointRights of addedOrganizationUnitDefaultAccessControlPointRights) {
            for (const acpdr of unitDefaultAccessControlPointRights.defaultAccessControlPointRights) {
                let alreadyAddedAccessRight = request.accessRights.find((f) => f.accessControlPointId === acpdr.accessControlPointId);
                if (alreadyAddedAccessRight) {
                    alreadyAddedAccessRight.access = alreadyAddedAccessRight.access || acpdr.defaultPrivileges.access;
                    alreadyAddedAccessRight.read = alreadyAddedAccessRight.read || acpdr.defaultPrivileges.read;
                    alreadyAddedAccessRight.remoteAccess = alreadyAddedAccessRight.remoteAccess || acpdr.defaultPrivileges.remoteAccess;
                    alreadyAddedAccessRight.grant = alreadyAddedAccessRight.grant || acpdr.defaultPrivileges.grant;
                    alreadyAddedAccessRight.config = alreadyAddedAccessRight.config || acpdr.defaultPrivileges.config;
                    alreadyAddedAccessRight.snapshot = alreadyAddedAccessRight.snapshot || acpdr.defaultPrivileges.snapshot;
                }
                else {
                    request.accessRights.push({
                        accessControlPointId: acpdr.accessControlPointId,
                        access: acpdr.defaultPrivileges.access,
                        read: acpdr.defaultPrivileges.read,
                        remoteAccess: acpdr.defaultPrivileges.remoteAccess,
                        grant: acpdr.defaultPrivileges.grant,
                        config: acpdr.defaultPrivileges.config,
                        snapshot: acpdr.defaultPrivileges.snapshot,
                    });
                }
            }
        }
    }
    if (request.credentials) {
        for (let i = 0; i < request.credentials.length; i++) {
            if (request.credentials[i].type == app_enums_1.enums.CredentialType.VehiclePlate) {
                request.credentials[i].data = request.credentials[i].data.toUpperCase();
            }
        }
    }
    const hook = business_hooks_1.armonHookManager.getBeforeUserInsertHook(organizationId);
    if (hook) {
        const res = await hook({
            identity: request,
            organizationId: organizationId,
            requesterUserId: authResult.userId,
        });
        if (!res.isValid) {
            throw (0, api_error_1.generateTranslatedError)(res.errorCode ?? app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR, res.errorMessage, res.replacements);
        }
    }
    const result = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const licenceSettings = await dal_manager_1.dbManager.accessOrganization.getModuleLicenceSettings(organizationId, restapi_1.ArmonApplicationModule.Base, trx);
        if (!licenceSettings) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR, "ERRORS.GENERAL.MODULENOTFOUND");
        }
        else {
            if (licenceSettings.licenseControlPolicy === dal_constants_1.DalConstants.LicenseControlPolicy.Instant) {
                const licenceCheckResult = await dal_manager_1.dbManager.accessOrganization.baseModuleLicenceVacancyCheck(organizationId, licenceSettings, trx);
                if (!licenceCheckResult) {
                    (0, business_notification_1.sendOrganizationModuleLicenseLimitReachedNotification)({ organizationId, applicationModule: dal_constants_1.DalConstants.ArmonApplicationModule.Base });
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.PAYMENT_REQUIRED, "ERRORS.IDENTITY.BASE_MODULE_LICENCE_EXCEEDED", null, true, true);
                }
            }
        }
        return dal_manager_1.dbManager.accessIdentity.insertIdentity(organizationId, authResult.userId, {
            insertIdentity: request,
            notifyUsersForChange: true,
        }, trx);
    }, authResult.userId, organizationId);
    const afterHook = business_hooks_1.armonHookManager.getAfterIdentityUpsertHook(organizationId);
    if (afterHook) {
        await afterHook(null, result);
    }
    if (request.userAccount?.username) {
        (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId).then((organizationSettings) => {
            if (organizationSettings.notification.enableSendingCreatedAccountEmail && new RegExp(utils_1.emailRegex).exec(request.userAccount.username)) {
                (0, business_notification_1.sendAccountCreatedNotification)(organizationId, { username: request.userAccount.username, password: request.userAccount.password, accountOwnerUserId: result.id });
            }
        });
    }
    res.json(result);
});
module.exports.updateIdentity = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let request = new api_user_v1_models_1.RestUserApiV1.UpdateIdentityRequest().validate(req.body, new api_validatorhelper_1.ValidatorHelper());
    if ((0, api_util_2.hasDuplicateUuids)([...request.addedOrganizationUnits.map((unit) => unit.organizationUnitId), ...request.updatedOrganizationUnits.map((unit) => unit.organizationUnitId)])) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.DUPLICATEORGANIZATIONUNITS");
    }
    if ((0, api_util_2.hasDuplicateUuids)(request.addedUserGroupIds)) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.DUPLICATEUSERGROUPS");
    }
    let authResult = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    const { r: targetUserRoleId } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: request.id });
    if (targetUserRoleId != request.organizationProfile.roleId) {
        const { r: requesterRoleId } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
        const canChangeRole = await (0, business_roles_1.canUserChangeRole)(organizationId, requesterRoleId, request.organizationProfile.roleId, request.id);
        if (!canChangeRole) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.INVALIDROLEDEF");
        }
    }
    let canChange = await (0, business_roles_1.canChangeAccessRights)(organizationId, request.id, request.updatedAccessRights, request.removedAccessControlPointIds);
    if (!canChange) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.UNABLETOCHANGEACCESSRIGHTLASTGRANT");
    }
    if (request.addedAccessRights?.length > 0 || request.updatedAccessRights?.length > 0 || request.removedAccessControlPointIds?.length > 0) {
        let userAccessRights = await dal_manager_1.dbManager.accessAccessControlPoint.getUserAccessControlPointRights(organizationId, authResult.jwt.userId);
        if (request.addedAccessRights.length > 0 && !userAccessRights.some((u) => request.addedAccessRights.some((r) => r.accessControlPointId == u.accessControlPointId) && u.grant)) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.UNABLETOCHANGEACCESSRIGHT");
        }
        if (request.updatedAccessRights.length > 0 && !userAccessRights.some((u) => request.updatedAccessRights.some((r) => r.accessControlPointId == u.accessControlPointId) && u.grant)) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.UNABLETOCHANGEACCESSRIGHT");
        }
        if (request.removedAccessControlPointIds.length > 0 && !userAccessRights.some((u) => request.removedAccessControlPointIds.some((r) => r == u.accessControlPointId) && u.grant)) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.UNABLETOCHANGEACCESSRIGHT");
        }
    }
    if ((request.addedOrganizationUnits?.length && request.addDefaultAccessRightsOfUnits) || (request.removedOrganizationUnitIds?.length && request.removeDefaultAccessRightsOfUnits)) {
        let currentIdentity = null;
        let addedOrganizationUnitDefaultAccessControlPointRights = null;
        let removedOrganizationUnitDefaultAccessControlPointRights = null;
        let remainingOrganizationUnitDefaultAccessControlPointRights = null;
        await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            currentIdentity = await dal_manager_1.dbManager.accessIdentity.getIdentityDetailed(organizationId, authResult.jwt.userId, {
                userId: request.id,
                hasOrganizationWideRight: true,
                hasIdentityFullWrite: false,
                permittedUnitIds: null,
                hasOrganizationWideRightToSeePassiveUsers: false,
            });
            addedOrganizationUnitDefaultAccessControlPointRights = await dal_manager_1.dbManager.accessOrganizationUnit.getDefaultAccessControlPointRightsOfOrganizationUnits(organizationId, request.addedOrganizationUnits?.map((m) => m.organizationUnitId) ?? [], trx);
            removedOrganizationUnitDefaultAccessControlPointRights = await dal_manager_1.dbManager.accessOrganizationUnit.getDefaultAccessControlPointRightsOfOrganizationUnits(organizationId, request.removedOrganizationUnitIds ?? [], trx);
            remainingOrganizationUnitDefaultAccessControlPointRights = await dal_manager_1.dbManager.accessOrganizationUnit.getDefaultAccessControlPointRightsOfOrganizationUnits(organizationId, currentIdentity.organizationUnits?.map((m) => m.organizationUnitId).filter((f) => !request.removedOrganizationUnitIds.includes(f)) ?? [], trx);
        }, authResult.jwt.userId, organizationId);
        if (request.addedOrganizationUnits?.length > 0 && request.addDefaultAccessRightsOfUnits) {
            const accessRightsToAdd = [];
            const accessRightsToUpdate = [];
            for (const unitDefaultAccessRight of addedOrganizationUnitDefaultAccessControlPointRights) {
                for (const acpDefaultRights of unitDefaultAccessRight.defaultAccessControlPointRights) {
                    let arToUpdate = currentIdentity.accessRights.find((f) => f.accessControlPointId === acpDefaultRights.accessControlPointId);
                    if (arToUpdate) {
                        accessRightsToUpdate.push({
                            accessControlPointId: arToUpdate.accessControlPointId,
                            access: arToUpdate.access || acpDefaultRights.defaultPrivileges.access,
                            read: arToUpdate.read || acpDefaultRights.defaultPrivileges.read,
                            grant: arToUpdate.grant || acpDefaultRights.defaultPrivileges.grant,
                            snapshot: arToUpdate.snapshot || acpDefaultRights.defaultPrivileges.snapshot,
                            config: arToUpdate.config || acpDefaultRights.defaultPrivileges.config,
                            remoteAccess: arToUpdate.remoteAccess || acpDefaultRights.defaultPrivileges.remoteAccess,
                        });
                    }
                    else {
                        accessRightsToAdd.push({
                            accessControlPointId: acpDefaultRights.accessControlPointId,
                            access: acpDefaultRights.defaultPrivileges.access,
                            read: acpDefaultRights.defaultPrivileges.read,
                            grant: acpDefaultRights.defaultPrivileges.grant,
                            snapshot: acpDefaultRights.defaultPrivileges.snapshot,
                            config: acpDefaultRights.defaultPrivileges.config,
                            remoteAccess: acpDefaultRights.defaultPrivileges.remoteAccess,
                        });
                    }
                }
            }
            if (!request.updatedAccessRights) {
                request.updatedAccessRights = [];
            }
            if (!request.addedAccessRights) {
                request.addedAccessRights = [];
            }
            if (!request.removedAccessControlPointIds) {
                request.removedAccessControlPointIds = [];
            }
            for (const artu of accessRightsToUpdate) {
                const alreadyUpdatedAccessRight = request.updatedAccessRights.find((f) => f.accessControlPointId === artu.accessControlPointId);
                if (alreadyUpdatedAccessRight) {
                    alreadyUpdatedAccessRight.access = alreadyUpdatedAccessRight.access || artu.access;
                    alreadyUpdatedAccessRight.read = alreadyUpdatedAccessRight.read || artu.read;
                    alreadyUpdatedAccessRight.grant = alreadyUpdatedAccessRight.grant || artu.grant;
                    alreadyUpdatedAccessRight.snapshot = alreadyUpdatedAccessRight.snapshot || artu.snapshot;
                    alreadyUpdatedAccessRight.config = alreadyUpdatedAccessRight.config || artu.config;
                    alreadyUpdatedAccessRight.remoteAccess = alreadyUpdatedAccessRight.remoteAccess || artu.remoteAccess;
                }
                else {
                    request.updatedAccessRights.push(artu);
                }
            }
            for (const arta of accessRightsToAdd) {
                let alreadyAddedAccessRight = request.addedAccessRights.find((f) => f.accessControlPointId === arta.accessControlPointId);
                if (alreadyAddedAccessRight) {
                    alreadyAddedAccessRight.access = alreadyAddedAccessRight.access || arta.access;
                    alreadyAddedAccessRight.read = alreadyAddedAccessRight.read || arta.read;
                    alreadyAddedAccessRight.grant = alreadyAddedAccessRight.grant || arta.grant;
                    alreadyAddedAccessRight.snapshot = alreadyAddedAccessRight.snapshot || arta.snapshot;
                    alreadyAddedAccessRight.config = alreadyAddedAccessRight.config || arta.config;
                    alreadyAddedAccessRight.remoteAccess = alreadyAddedAccessRight.remoteAccess || arta.remoteAccess;
                }
                else {
                    request.addedAccessRights.push(arta);
                }
            }
        }
        if (request.removedOrganizationUnitIds?.length > 0 && request.removeDefaultAccessRightsOfUnits) {
            for (const unitDefaultAccessRight of removedOrganizationUnitDefaultAccessControlPointRights) {
                for (const acpDefaultRights of unitDefaultAccessRight.defaultAccessControlPointRights) {
                    let shouldRemoveRight = true;
                    for (const roudacpr of remainingOrganizationUnitDefaultAccessControlPointRights) {
                        if (roudacpr.defaultAccessControlPointRights.map((m) => m.accessControlPointId).includes(acpDefaultRights.accessControlPointId)) {
                            shouldRemoveRight = false;
                        }
                    }
                    if (request.addedAccessRights.map((m) => m.accessControlPointId).includes(acpDefaultRights.accessControlPointId) ||
                        request.updatedAccessRights.map((m) => m.accessControlPointId).includes(acpDefaultRights.accessControlPointId)) {
                        shouldRemoveRight = false;
                    }
                    if (shouldRemoveRight &&
                        currentIdentity.accessRights.map((m) => m.accessControlPointId).includes(acpDefaultRights.accessControlPointId) &&
                        !request.removedAccessControlPointIds.includes(acpDefaultRights.accessControlPointId)) {
                        request.removedAccessControlPointIds.push(acpDefaultRights.accessControlPointId);
                    }
                }
            }
        }
    }
    if (request.addedCredentials) {
        for (let i = 0; i < request.addedCredentials.length; i++) {
            if (request.addedCredentials[i].type == app_enums_1.enums.CredentialType.VehiclePlate) {
                request.addedCredentials[i].data = request.addedCredentials[i].data.toUpperCase();
            }
        }
    }
    if (request.removedOrganizationUnitIds && request.removedOrganizationUnitIds.length > 0) {
        const orgUnitWriteAuthResult = (0, api_securityhelper_1.authorizeForUserHierarchy)(req, organizationId);
        if (!orgUnitWriteAuthResult.hasOrganizationWide) {
            let permittedOrganizationUnitIds = [];
            await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                let units = (await dal_manager_1.dbManager.accessOrganizationUnit.getAllSuccessorOrganizationUnitsBasic(organizationId, orgUnitWriteAuthResult.permittedUnitIds, trx)).map((organizationUnit) => organizationUnit.id);
                permittedOrganizationUnitIds.push(...units, ...orgUnitWriteAuthResult.permittedUnitIds);
                permittedOrganizationUnitIds = [...new Set(permittedOrganizationUnitIds)];
                if (request.removedOrganizationUnitIds.some((rouId) => !permittedOrganizationUnitIds.includes(rouId))) {
                    console.log("Cannot remove! not permitted for this unit!");
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.CANNOTREMOVEFROMHIERARCHY");
                }
                const canRemove = await dal_manager_1.dbManager.accessUser.canManagerRemoveUserFromUnits({
                    organizationId: organizationId,
                    managerId: authResult.jwt.userId,
                    removedOrganizationUnitIds: request.removedOrganizationUnitIds,
                    addedOrganizationUnitIds: request.addedOrganizationUnits.map((aou) => aou.organizationUnitId),
                    userId: request.id,
                });
                if (!canRemove) {
                    throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.CONFLICT, "ERRORS.GENERAL.CANNOTREMOVEFROMHIERARCHY");
                }
            }, orgUnitWriteAuthResult.jwt.userId, organizationId);
        }
    }
    const hook = business_hooks_1.armonHookManager.getBeforeUserUpdateHook(organizationId);
    if (hook) {
        const res = await hook({
            identity: request,
            organizationId: organizationId,
            requesterUserId: authResult.jwt.userId,
        });
        if (!res.isValid) {
            throw (0, api_error_1.generateTranslatedError)(res.errorCode ?? app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR, res.errorMessage, res.replacements);
        }
    }
    const { oldIdentity, newIdentity } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessIdentity.updateIdentity(organizationId, authResult.jwt.userId, trx, request, true);
    }, authResult.jwt.userId, organizationId);
    if (!oldIdentity.organizationProfile.isDisabled && newIdentity.organizationProfile.isDisabled) {
        if ((0, api_arvento_1.isArventoIntegrationEnabled)(organizationId)) {
            (0, api_arvento_1.sendUserDisabledRequest)(newIdentity.organizationProfile.uniqueId, organizationId);
        }
    }
    const afterHook = business_hooks_1.armonHookManager.getAfterIdentityUpsertHook(organizationId);
    if (afterHook) {
        await afterHook(oldIdentity, newIdentity);
    }
    if ((request.addedAccessRights && request.addedAccessRights.find((a) => a.remoteAccess)) ||
        (request.updatedAccessRights && request.updatedAccessRights.find((a) => a.remoteAccess)) ||
        (request.removedAccessControlPointIds && request.removedAccessControlPointIds.length > 0)) {
        await app_httpserver_1.httpServer.sendRemoteAccessGrantsChangedNotification(organizationId, request.id);
    }
    if (request.userAccount?.username) {
        (0, dal_memcache_1.getCacheOrganizationSettings)(organizationId).then((organizationSettings) => {
            if (organizationSettings.notification.enableSendingCreatedAccountEmail && new RegExp(utils_1.emailRegex).exec(request.userAccount.username)) {
                (0, business_notification_1.sendAccountCreatedNotification)(organizationId, { username: request.userAccount.username, password: request.userAccount.password, accountOwnerUserId: request.id });
            }
        });
    }
    res.json(newIdentity);
});
module.exports.collectUserIdsByFieldSelect = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let request = new api_user_v1_models_1.RestUserApiV1.CollectUserIdsByFieldSelectRequest(req.body, new api_validatorhelper_1.ValidatorHelper().validateUUID("oid", false, req.swagger.params.oid.value));
    let organizationId = req.swagger.params.oid.value;
    let jwt = (0, api_securityhelper_1.authorizeForCollectUserIdsByFieldSelect)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessIdentity.collectUserIdsByFieldSelect(organizationId, {
        fieldName: request.fieldName,
        fieldType: request.fieldType,
        isExtensionField: request.isExtensionField,
        parameters: request.parameters,
    });
    res.json(dbResult);
});
module.exports.getIdentityDetailedById = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "userId");
    let authResult = (0, api_securityhelper_1.authorizeForGetIdentityDetailed)(req, organizationId);
    let hasFullWrite = (0, api_securityhelper_1.hasIdentityFullWrite)(req, organizationId);
    try {
        let result = await dal_manager_1.dbManager.accessIdentity.getIdentityDetailed(organizationId, authResult.jwt.userId, {
            userId,
            hasOrganizationWideRight: authResult.hasOrganizationWide,
            hasIdentityFullWrite: hasFullWrite,
            permittedUnitIds: authResult.permittedUnitIds,
            hasOrganizationWideRightToSeePassiveUsers: authResult.jwt.isPermittedForOrganization(organizationId, predefined_permissions_1.Permissions.system.getOrganizationWidePassiveUsersView()),
        });
        res.json(result);
    }
    catch (error) {
        throw (0, api_error_1.generateTranslatedError)(error.errorCode, error.phrase, error.replacements, error.noReport, error.showAsModal);
    }
});
module.exports.getIdentityCard = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userId = (0, api_securityhelper_1.getUserIdFromToken)(req);
    try {
        let result = await dal_manager_1.dbManager.accessUser.getUserCard(organizationId, userId);
        res.json(result);
    }
    catch (error) {
        throw (0, api_error_1.generateApiErrorWithCode)(error.code, error.message, error.url);
    }
});
module.exports.getIdentityUserActions = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "userId");
    let authResult = (0, api_securityhelper_1.authorizeForGetIdentityDetailed)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.accessLog.listUserActionHistory(organizationId, userId, "id", app_enums_1.enums.UserActionCategory.Main, [
        app_enums_1.enums.UserMainActionType.AddAccessRight,
        app_enums_1.enums.UserMainActionType.AddCredential,
        app_enums_1.enums.UserMainActionType.InsertIdentity,
        app_enums_1.enums.UserMainActionType.RemoveAccessRight,
        app_enums_1.enums.UserMainActionType.RemoveCredential,
        app_enums_1.enums.UserMainActionType.UpdateIdentity,
        app_enums_1.enums.UserMainActionType.UpdateProfile,
        app_enums_1.enums.UserMainActionType.UpdateCredential,
        app_enums_1.enums.UserMainActionType.AssignUserToUserGroup,
        app_enums_1.enums.UserMainActionType.RemoveUserFromUserGroup,
        app_enums_1.enums.UserMainActionType.AssignUserToOrganizationUnit,
        app_enums_1.enums.UserMainActionType.RemoveUserFromOrganizationUnit,
        app_enums_1.enums.UserMainActionType.UpdateUserOrganizationUnitRole,
        app_enums_1.enums.UserMainActionType.AccountDeleted,
        app_enums_1.enums.UserMainActionType.UsernameUpdated,
        app_enums_1.enums.UserMainActionType.AccountCreated,
    ]);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    let roles = await dal_manager_1.dbManager.accessUser.listRolesOfOrganization(organizationId);
    let result = (0, business_main_1.transformMainActionItemsLocalizedMessages)(dbResult, locale, roles);
    res.json({
        items: result,
    });
});
module.exports.addIdentityToForbiddenList = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let authResult = (0, api_securityhelper_1.authorizeForIdentityForbidEdit)(req, organizationId);
    req.body.userId = userId;
    let request = new api_user_v1_models_1.RestUserApiV1.AddForbiddenStatusRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessIdentity.upsertIdentityForbiddenStatus(organizationId, authResult.jwt.userId, request);
    if (dbResult) {
    }
    res.json(dbResult);
});
module.exports.updateForbidStatus = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let forbidId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.fid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForIdentityForbidEdit)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.UpdateForbiddenStatusRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.accessIdentity.upsertIdentityForbiddenStatus(organizationId, authResult.jwt.userId, lodash_1.default.assign({}, request, { id: forbidId, userId: null }));
    let userForbiddance = await dal_manager_1.dbManager.accessIdentity.getForbiddenStatus(organizationId, forbidId);
    if (userForbiddance) {
    }
    res.json();
});
module.exports.deleteForbidStatus = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let forbidStatusId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.fid.value, "forbidStatusId");
    let authResult = (0, api_securityhelper_1.authorizeForIdentityForbidEdit)(req, organizationId);
    let userForbiddance = await dal_manager_1.dbManager.accessIdentity.getForbiddenStatus(organizationId, forbidStatusId);
    let dbResult = await dal_manager_1.dbManager.accessIdentity.deleteIdentityForbiddenStatus(organizationId, authResult.jwt.userId, forbidStatusId);
    if (!dbResult) {
        throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.NOT_FOUND, "ERRORS.GENERAL.NOTFOUND");
    }
    if (userForbiddance) {
    }
    res.json();
});
module.exports.listForbiddenUsers = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForIdentityForbidRead)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.ForbiddenListRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    request.hasOrganizationWideRight = authResult.hasOrganizationWide;
    request.permittedUnitIds = authResult.permittedUnitIds;
    const listForbidden = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const forbiddances = await dal_manager_1.dbManager.accessIdentity.listForbidden({
            organizationId,
            filter: request,
            requesterUserId: authResult.jwt.userId,
            trx,
        });
        const basicUserInfosOfForbiddenUsers = await dal_manager_1.dbManager.accessUser.getBasicUserInfoPg({ organizationId, userIds: forbiddances.items.map((i) => i.userId), trx });
        const forbiddenItems = forbiddances.items.map((i) => {
            const basicUserInfo = basicUserInfosOfForbiddenUsers.find((b) => b.id === i.userId);
            return {
                id: i.id,
                user: basicUserInfo,
                startUtc: i.startUtc,
                endUtc: i.endUtc,
                regionId: i.regionId,
                regionName: i.regionName,
                note: i.note,
                credentialIds: i.credentialIds,
                userCaptions: basicUserInfo.captionLines,
            };
        });
        const result = {
            pagination: forbiddances.pagination,
            items: forbiddenItems,
        };
        return result;
    }, authResult.jwt.userId, organizationId);
    res.json(listForbidden);
});
module.exports.listForbiddenUsersForExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let jwt = req.auth_token;
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForIdentityForbidRead)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.ForbiddenListRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    request.hasOrganizationWideRight = authResult.hasOrganizationWide;
    request.permittedUnitIds = authResult.permittedUnitIds;
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.ForbiddenUsers,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getForbidStatusHistory = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForIdentityForbidEdit)(req, organizationId);
    const fid = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.fid.value, "forbidId");
    const dbResult = await dal_manager_1.dbManager.accessLog.listUserActionHistory(organizationId, fid, "fid", app_enums_1.enums.UserActionCategory.Main, [
        app_enums_1.enums.UserMainActionType.AddForbiddenState,
        app_enums_1.enums.UserMainActionType.UpdateForbiddenState,
        app_enums_1.enums.UserMainActionType.RemoveForbiddenState,
    ]);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.jwt.userId });
    const result = (0, business_main_1.transformMainActionItemsLocalizedMessages)(dbResult, locale);
    res.json({
        items: result,
    });
});
module.exports.changePasswordOfIdentity = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let identityChangePassword = new api_user_v1_models_1.RestUserApiV1.ChangePasswordOfIdentity(req.body, new api_validatorhelper_1.ValidatorHelper().validateUUID("oid", false, req.swagger.params.oid.value));
    let organizationId = req.swagger.params.oid.value;
    let authResult = (0, api_securityhelper_1.authorizeForIdentityAccountUpdate)(req, organizationId);
    await dal_manager_1.dbManager.accessUser.changePasswordOfIdentity(organizationId, identityChangePassword, authResult.hasOrganizationWide, authResult.jwt);
    res.json({});
});
module.exports.changeMyPassword = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let changeMyPasswordRequest = new api_user_v1_models_1.RestUserApiV1.ChangeMyPasswordRequest(req.body, new api_validatorhelper_1.ValidatorHelper().validateUUID("oid", false, req.swagger.params.oid.value));
    let organizationId = req.swagger.params.oid.value;
    let userId = (0, api_securityhelper_1.getUserIdFromToken)(req);
    await dal_manager_1.dbManager.accessUser.changeMyPassword(organizationId, userId, changeMyPasswordRequest);
    res.json({});
});
module.exports.addAccessRightForUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const authResult = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    const addAccessRightRequest = new api_user_v1_models_1.RestUserApiV1.AddAccessRightForUserRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const dbResult = await dal_manager_1.dbManager.accessIdentity.addAccessRightForUser(organizationId, authResult.jwt.userId, trx, {
            userId,
            accessRights: addAccessRightRequest.addedAccessRights,
        });
        if (dbResult) {
            res.json({
                userId,
                accessRights: await dal_manager_1.dbManager.accessIdentity.getUserAccessRights({ organizationId, userId, trx }),
            });
        }
        else {
            res.json(dbResult);
        }
    }, userId, organizationId);
});
module.exports.updateUsername = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const updateUsernameRequest = new api_user_v1_models_1.RestUserApiV1.UpdateUsernameRequest(req.body, new api_validatorhelper_1.ValidatorHelper().validateUUID("oid", false, req.swagger.params.oid.value).validateUUID("uid", false, req.swagger.params.uid.value));
    const organizationId = req.swagger.params.oid.value;
    const userId = req.swagger.params.uid.value;
    const auth = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const { oldUsername } = await dal_manager_1.dbManager.accessIdentity.updateUsername(organizationId, userId, updateUsernameRequest.username, trx);
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItemPg(organizationId, auth.jwt.userId, trx, {
            oId: organizationId,
            o: auth.jwt.userId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Main,
            tg: ["Identity", "Username"],
            t: dal_constants_1.DalConstants.UserMainActionType.UsernameUpdated,
            d: {
                id: userId,
                c: JSON.stringify([
                    {
                        f: "username",
                        o: oldUsername,
                        n: updateUsernameRequest.username,
                        i: true,
                    },
                ]),
            },
        });
    }, auth.jwt.userId, organizationId);
    res.sendStatus(terminal_1.HttpStatusCode.OK);
});
module.exports.deleteUserAccount = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    new api_validatorhelper_1.ValidatorHelper().validateUUID("oid", false, req.swagger.params.oid.value).validateUUID("uid", false, req.swagger.params.uid.value);
    const organizationId = req.swagger.params.oid.value;
    const userId = req.swagger.params.uid.value;
    const auth = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        const { deletedUsername } = await dal_manager_1.dbManager.accessIdentity.deleteUserAccount(organizationId, userId, trx);
        await dal_manager_1.dbManager.accessLog.addUserActionHistoryItemPg(organizationId, auth.jwt.userId, trx, {
            oId: organizationId,
            o: auth.jwt.userId,
            u: new Date(),
            c: dal_constants_1.DalConstants.UserActionCategory.Main,
            tg: ["Identity", "Username"],
            t: dal_constants_1.DalConstants.UserMainActionType.AccountDeleted,
            d: {
                id: userId,
                c: JSON.stringify([
                    {
                        f: "username",
                        o: deletedUsername,
                        n: null,
                        i: true,
                    },
                ]),
            },
        });
    }, auth.jwt.userId, organizationId);
    res.sendStatus(terminal_1.HttpStatusCode.OK);
});
module.exports.clearUserMobileInstallIdentifiers = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForIdentityAccountUpdate)(req, organizationId);
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        await dal_manager_1.dbManager.accessOAuth.clearMobileInstallIdentifiers(organizationId, userId, trx);
    }, auth.jwt.userId, organizationId);
    res.send(200);
});
