"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_logs_1 = require("../../../../app.logs");
const report_util_1 = require("../../../../business/report/report.util");
const dal_manager_1 = require("../../../../dal/dal.manager");
const predefined_permissions_1 = require("../../../../dal/db/predefined/predefined.permissions");
const messageBroker_server_to_report_pub_1 = require("../../../../messageBroker/messageBroker.server-to-report.pub");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_user_v1_models_1 = require("../api.user.v1.models");
module.exports.searchCredentials = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    const request = new api_user_v1_models_1.RestUserApiV1.SearchCredentialRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (!authResult.hasOrganizationWide && !request.organizationUnitIds?.length) {
        request.organizationUnitIds = authResult.jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.identity.getReadBasic());
        request.filterUsersHierarchically = true;
    }
    const response = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessIdentity.searchCredentials(organizationId, authResult.jwt.userId, request, trx);
    }, authResult.jwt.userId, organizationId);
    res.json(response);
});
module.exports.searchCredentialsForExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForSearchIdentity)(req, organizationId);
    let request = new api_user_v1_models_1.RestUserApiV1.SearchCredentialRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (!authResult.hasOrganizationWide && !request.organizationUnitIds?.length) {
        request.organizationUnitIds = authResult.jwt.getPermittedUnitIdsFor(organizationId, predefined_permissions_1.Permissions.identity.getReadBasic());
        request.filterUsersHierarchically = true;
    }
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.Credential,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: authResult.jwt.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.addCredential = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    const { credentials } = new api_user_v1_models_1.RestUserApiV1.AssignCredentialToUser(req.body, new api_validatorhelper_1.ValidatorHelper());
    const addedCredetial = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessIdentity.assignCredentialToUser(organizationId, auth.jwt.userId, trx, { credentials, userId });
    }, auth.jwt.userId, organizationId);
    res.json(addedCredetial);
});
module.exports.removeCredential = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const auth = (0, api_securityhelper_1.authorizeForUpdateIdentity)(req, organizationId);
    const { credentialIds } = new api_user_v1_models_1.RestUserApiV1.UnassignCredentialToUser(req.body, new api_validatorhelper_1.ValidatorHelper());
    const removedCredentialIds = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessIdentity.unAssignCredentialFromUser(organizationId, auth.jwt.userId, trx, { credentialIds, userId });
    }, auth.jwt.userId, organizationId);
    res.json(removedCredentialIds);
});
