"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const luxon_1 = require("luxon");
const app_config_1 = require("../../../../app.config");
const app_enums_1 = require("../../../../app.enums");
const app_filesystem_1 = require("../../../../app.filesystem");
const app_logs_1 = require("../../../../app.logs");
const file_type_1 = __importDefault(require("file-type"));
const business_common_1 = require("../../../../business/business.common");
const business_hooks_1 = require("../../../../business/business.hooks");
const business_pacs_1 = require("../../../../business/pacs/business.pacs");
const business_pacs_claim_1 = require("../../../../business/pacs/business.pacs.claim");
const business_pacs_ppermission_1 = require("../../../../business/pacs/business.pacs.ppermission");
const business_pacs_validation_1 = require("../../../../business/pacs/business.pacs.validation");
const report_util_1 = require("../../../../business/report/report.util");
const dal_access_error_1 = require("../../../../dal/access/dal.access.error");
const dal_constants_1 = require("../../../../dal/dal.constants");
const dal_manager_1 = require("../../../../dal/dal.manager");
const messageBroker_server_to_report_pub_1 = require("../../../../messageBroker/messageBroker.server-to-report.pub");
const api_error_1 = require("../../../api.error");
const api_securityhelper_1 = require("../../../api.securityhelper");
const api_util_1 = require("../../../api.util");
const api_validatorhelper_1 = require("../../../api.validatorhelper");
const api_pacs_v1_models_1 = require("../api.pacs.v1.models");
const moment = require("moment");
module.exports.upsertPPermissionType = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionTypeRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    request.type = app_enums_1.enums.PredefinedPPermissionType.Other;
    const dbResult = await dal_manager_1.dbManager.organizationTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.upsertPPermissionTypePg({ organizationId, args: request, trx });
    }, authResult.userId, organizationId);
    res.json({
        id: dbResult,
    });
});
module.exports.getPPermissionType = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let ppermissionTypeId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionTypeId");
    let dbResult = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermissionTypeId);
    res.json(dbResult);
});
module.exports.removePPermissionType = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let ppermissionTypeId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionTypeId");
    let dbResult = await dal_manager_1.dbManager.accessPacs.removePPermissionType(organizationId, ppermissionTypeId);
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "Could not remove ppermission type" }));
    }
    res.json();
});
module.exports.listPPermissionTypes = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const hrReadAuth = (0, api_securityhelper_1.hasHRRead)(req, organizationId);
    const unitManagerReadPermissionStatus = (0, api_securityhelper_1.hasPermissionReadForAnyUnit)(req, organizationId);
    const request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionTypeFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.accessPacs.listPPermissionTypes(organizationId, {
        name: request.name,
        isPassive: request.isPassive,
        take: request.pagination.take,
        skip: request.pagination.skip,
        hrStatus: hrReadAuth,
        unitManagerReadPermissionStatus,
    });
    res.json(dbResult);
});
module.exports.upsertSelfPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let request = new api_pacs_v1_models_1.RestPacsApiV1.SelfPPermissionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (moment(request.startDateTime).isSameOrAfter(moment(request.endDateTime))) {
        (0, dal_access_error_1.throwDbAccessBadRequestError)("Permission start time cannot be same as or after end time");
    }
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: jwt.userId });
    let ppermissionValidator = new business_pacs_validation_1.ValidateUserPPermission();
    let validationResult = await ppermissionValidator.validateUserPPermission(organizationId, request.ppermissionTypeId, jwt.userId, locale, new Date(request.startDateTime), new Date(request.endDateTime), request.id);
    if (!validationResult.isOk)
        (0, dal_access_error_1.throwDbAccessConflictError)(validationResult.errorMessage);
    const formValidation = await business_common_1.BusinessCommon.validateDynamicForm({
        organizationId: organizationId,
        dynamicFormId: request.dynamicFormId,
        dynamicFormData: request.dynamicFormData,
    });
    if (formValidation?.length > 0) {
        let errors = "Error(s): ";
        for (const err of formValidation) {
            errors += err.message;
        }
        (0, dal_access_error_1.throwDbAccessConflictError)(errors);
    }
    const ppermissionId = await dal_manager_1.dbManager.accessPacs.upsertSelfPPermission(organizationId, {
        id: request.id,
        ppermissionTypeId: request.ppermissionTypeId,
        startDateTime: request.startDateTime,
        endDateTime: request.endDateTime,
        note: request.note,
        attachmentList: request.attachmentList,
        userAndDeputyList: [
            {
                userId: jwt.userId,
                deputyUserId: request.deputyUserId,
            },
        ],
        requestUserId: jwt.userId,
        dynamicFormId: request.dynamicFormId,
        dynamicFormData: request.dynamicFormData,
    });
    (0, business_pacs_ppermission_1.sendNextPPermissionNotification)({ organizationId, ppermissionId, userIds: [jwt.userId], requesterUserId: jwt.userId });
    res.json({});
});
module.exports.upsertPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (moment(request.startDateTime).isSameOrAfter(moment(request.endDateTime))) {
        (0, dal_access_error_1.throwDbAccessBadRequestError)("Permission start time cannot be same as or after end time");
    }
    const formValidation = await business_common_1.BusinessCommon.validateDynamicForm({
        organizationId: organizationId,
        dynamicFormId: request.dynamicFormId,
        dynamicFormData: request.dynamicFormData,
    });
    if (formValidation?.length > 0) {
        let errors = "Error(s): ";
        for (const err of formValidation) {
            errors += err.message;
        }
        (0, dal_access_error_1.throwDbAccessConflictError)(errors);
    }
    let dbResult = await dal_manager_1.dbManager.accessPacs.upsertPPermission(organizationId, {
        id: request.id,
        ppermissionTypeId: request.ppermissionTypeId,
        startDateTime: request.startDateTime,
        endDateTime: request.endDateTime,
        note: request.note,
        attachmentList: request.attachmentList,
        requestUserId: authResult.userId,
        requestRole: app_enums_1.enums.PPermissionInsertRole.Manager,
        approve: request.approve,
        dynamicFormId: request.dynamicFormId,
        dynamicFormData: request.dynamicFormData,
    });
    res.json(dbResult);
});
module.exports.upsertPPermissionByHR = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    if (moment(request.startDateTime).isSameOrAfter(moment(request.endDateTime))) {
        (0, dal_access_error_1.throwDbAccessBadRequestError)("Permission start time cannot be same as or after end time");
    }
    const formValidation = await business_common_1.BusinessCommon.validateDynamicForm({
        organizationId: organizationId,
        dynamicFormId: request.dynamicFormId,
        dynamicFormData: request.dynamicFormData,
    });
    if (formValidation?.length > 0) {
        let errors = "Error(s): ";
        for (const err of formValidation) {
            errors += err.message;
        }
        (0, dal_access_error_1.throwDbAccessConflictError)(errors);
    }
    let dbResult = await dal_manager_1.dbManager.accessPacs.upsertPPermission(organizationId, {
        id: request.id,
        ppermissionTypeId: request.ppermissionTypeId,
        startDateTime: request.startDateTime,
        endDateTime: request.endDateTime,
        note: request.note,
        attachmentList: request.attachmentList,
        requestUserId: authResult.userId,
        requestRole: app_enums_1.enums.PPermissionInsertRole.HR,
        approve: request.approve,
        dynamicFormId: request.dynamicFormId,
        dynamicFormData: request.dynamicFormData,
    });
    res.json(dbResult);
});
module.exports.upsertPPermissionWithUsers = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionRequestWithUsers(req.body, new api_validatorhelper_1.ValidatorHelper());
    const formValidation = await business_common_1.BusinessCommon.validateDynamicForm({
        organizationId: organizationId,
        dynamicFormId: request.dynamicFormId,
        dynamicFormData: request.dynamicFormData,
    });
    if (formValidation?.length > 0) {
        let errors = "Error(s): ";
        for (const err of formValidation) {
            errors += err.message;
        }
        (0, dal_access_error_1.throwDbAccessConflictError)(errors);
    }
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    let validator = new business_pacs_validation_1.ValidateUserPPermission();
    let permValidation = await validator.validatePPermission(organizationId, request.ppermissionTypeId, request.userAndDeputyList?.map((ud) => ud.userId), locale, request.startDateTime, request.endDateTime, null, request.performValidations ?? true);
    if (permValidation.items && permValidation.items.length > 0 && !permValidation.items[0].isOk) {
        res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
            items: [
                {
                    error: permValidation.items[0].errorCode,
                    message: permValidation.items[0].errorMessage,
                    details: permValidation.items[0].userId,
                },
            ],
        });
    }
    else {
        let dbResult = await dal_manager_1.dbManager.accessPacs.upsertPPermissionWithUsers(organizationId, {
            id: request.id,
            ppermissionTypeId: request.ppermissionTypeId,
            startDateTime: request.startDateTime,
            endDateTime: request.endDateTime,
            userAndDeputyList: request.userAndDeputyList,
            note: request.note,
            attachmentList: request.attachmentList,
            requestUserId: authResult.userId,
            approve: request.approve,
            dynamicFormId: request.dynamicFormId,
            dynamicFormData: request.dynamicFormData,
        });
        res.json(dbResult);
    }
});
module.exports.insertMultiplePPermissionsForUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.MultiplePPermissionRequestForUser(req.body, new api_validatorhelper_1.ValidatorHelper());
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    for (const perm of request.ranges) {
        let oldPPermissions = await dal_manager_1.dbManager.accessPacs.listOldPPermissionsForValidation(organizationId, request.userAndDeputy.userId, request.ppermissionTypeId, perm.startDateTime, perm.endDateTime);
        if (oldPPermissions) {
            res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
                items: [
                    {
                        error: app_enums_1.enums.HttpStatusCode.CONFLICT,
                        message: i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONCONFLICTWITHSAMETYPE", locale: locale }),
                        details: luxon_1.DateTime.fromJSDate(perm.startDateTime).toLocaleString({
                            locale: locale == "en" ? "en-EN" : "tr-TR",
                        }) +
                            " - " +
                            luxon_1.DateTime.fromJSDate(perm.endDateTime).toLocaleString({
                                locale: locale == "en" ? "en-EN" : "tr-TR",
                            }),
                    },
                ],
            });
            return;
        }
    }
    let response = undefined;
    let existingUsage = await (0, business_pacs_claim_1.getPPermissionClaimAndUsage)(organizationId, request.userAndDeputy.userId, new Date(), request.ppermissionTypeId);
    let remainingManualAnnualPPermission;
    let organizationPACSModuleSettings;
    if (existingUsage.length > 0) {
        let holidays = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
        let unit = "";
        let totalUsage = 0;
        if (existingUsage[0].type.type === app_enums_1.enums.PredefinedPPermissionType.Annual) {
            organizationPACSModuleSettings = await dal_manager_1.dbManager.accessPacs.getOrganizationPACSModuleSettings(organizationId);
            if (organizationPACSModuleSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual) {
                remainingManualAnnualPPermission = (await dal_manager_1.dbManager.accessUser.getUserOrganizationProfileEmploymentInfo(organizationId, request.userAndDeputy.userId))
                    .manuallyRemainedAnnualPermission;
            }
            for (const range of request.ranges) {
                totalUsage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, existingUsage[0].type.id, range.startDateTime, range.endDateTime, holidays, request.userAndDeputy.userId);
                unit = "days";
            }
        }
        else {
            if (existingUsage[0].type.isDailyScheduled) {
                for (const range of request.ranges) {
                    totalUsage += await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, existingUsage[0].type.id, range.startDateTime, range.endDateTime, holidays, request.userAndDeputy.userId);
                    unit = "days";
                }
            }
            else {
                for (const range of request.ranges) {
                    totalUsage += await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, existingUsage[0].type.id, range.startDateTime, range.endDateTime, holidays);
                    unit = "minutes";
                }
            }
        }
        if (organizationPACSModuleSettings &&
            organizationPACSModuleSettings.ppermissionClaimProfile === app_enums_1.enums.PPermissionClaimProfile.Manual &&
            remainingManualAnnualPPermission !== null &&
            remainingManualAnnualPPermission !== undefined) {
            if (totalUsage > remainingManualAnnualPPermission) {
                res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
                    items: [
                        {
                            error: app_enums_1.enums.HttpStatusCode.CONFLICT,
                            message: i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONCLAIMERROR", locale: locale }),
                            details: "Required: " + totalUsage + ", Remaining: " + existingUsage[0].claim,
                        },
                    ],
                });
            }
            else {
                let ppermissionType = await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, request.ppermissionTypeId);
                if (!ppermissionType.approvementList || ppermissionType.approvementList.length < 1) {
                    await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUserMultiplePPermissions(organizationId, request.userAndDeputy.userId, request.ppermissionTypeId, totalUsage);
                }
                res.json(await dal_manager_1.dbManager.accessPacs.insertMultiplePPermissionsForUser(organizationId, {
                    ppermissionTypeId: request.ppermissionTypeId,
                    ranges: request.ranges,
                    userAndDeputy: request.userAndDeputy,
                    note: request.note,
                    attachmentList: request.attachmentList,
                    requestUserId: authResult.userId,
                }));
            }
        }
        else if (totalUsage > existingUsage[0].left) {
            res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
                items: [
                    {
                        error: app_enums_1.enums.HttpStatusCode.CONFLICT,
                        message: i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONCLAIMERROR", locale: locale }),
                        details: "Required: " + totalUsage + ", Remaining: " + existingUsage[0].claim,
                    },
                ],
            });
        }
        else {
            res.json(await dal_manager_1.dbManager.accessPacs.insertMultiplePPermissionsForUser(organizationId, {
                ppermissionTypeId: request.ppermissionTypeId,
                ranges: request.ranges,
                userAndDeputy: request.userAndDeputy,
                note: request.note,
                attachmentList: request.attachmentList,
                requestUserId: authResult.userId,
            }));
        }
    }
    else {
        throw (0, api_error_1.generateNotFoundApiError)({ message: "ppermission type not found" });
    }
});
module.exports.upsertPPermissionUsersBySelectionSession = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionWrite)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let userSelectionSessionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.sid.value, "userSelectionSessionId");
    let isHR = req.body.isHR;
    let approve = req.body.approve;
    if (isHR && !(0, api_securityhelper_1.hasHRApprovement)(req, organizationId)) {
        (0, dal_access_error_1.throwDbAccessUnauthorizedError)("PACS.PPERMISSION_EDIT_USERS_AUTHORIZATION_ERROR");
    }
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    let ppermissionValidator = new business_pacs_validation_1.ValidateUserPPermission();
    const validationResults = await ppermissionValidator.validatePPermissionByUserSelectionSessionId(organizationId, ppermissionId, userSelectionSessionId, locale);
    if (validationResults.length === 0) {
        let dbResult = await dal_manager_1.dbManager.accessPacs.upsertPPermissionUsersByUserSelectionSession(organizationId, ppermissionId, authResult.userId, userSelectionSessionId, approve, isHR ? dal_constants_1.DalConstants.PPermissionInsertRole.HR : dal_constants_1.DalConstants.PPermissionInsertRole.Manager);
        res.json(dbResult);
    }
    else {
        res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json(validationResults);
    }
});
module.exports.upsertDeputiesForPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authToken = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    const ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    const request = new api_pacs_v1_models_1.RestPacsApiV1.UserDeputyListRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.accessPacs.upsertPPermissionDeputyUsers(organizationId, ppermissionId, authToken.userId, request.list);
    res.json(dbResult);
});
module.exports.updatePPermissionDateRangeHR = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionWrite)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let request = new api_pacs_v1_models_1.RestPacsApiV1.DateRange(req.body, new api_validatorhelper_1.ValidatorHelper());
    await dal_manager_1.dbManager.accessPacs.updatePPermissionDateRangeByHR(organizationId, ppermissionId, authResult.userId, request.startDateTime, request.endDateTime);
    res.json();
});
module.exports.listPPermissionApprovementsForUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionRead)(req, organizationId);
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let dbResult = await dal_manager_1.dbManager.accessPacs.listPPermissionUserApprovementInfo(organizationId, ppermissionId, userId, authResult.userId);
    res.json({
        overallStatus: dbResult.overallStatus,
        items: dbResult.items.map((row) => {
            return {
                approverUsers: row.approverUsers.map((au) => {
                    return {
                        userCaptions: au.captionLines,
                        fullname: au.fullname,
                        uniqueId: au.uniqueId,
                        id: au.id,
                    };
                }),
                organizationUnitName: row.userOrganizationUnitName,
                canListerApprove: row.canListerApprove,
                isHR: row.isHR,
                order: row.order,
                status: row.status,
                approvementDate: row.approvementDate,
                note: row.note,
            };
        }),
    });
});
module.exports.listPPermissionApprovementsForUserForSelf = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let jwt = req.auth_token;
    let dbResult = await dal_manager_1.dbManager.accessPacs.listPPermissionUserApprovementInfo(organizationId, ppermissionId, userId, jwt.userId);
    res.json({
        overallStatus: dbResult.overallStatus,
        items: dbResult.items.map((row) => {
            return {
                approverUsers: row.approverUsers.map((au) => {
                    return {
                        userCaptions: au.captionLines,
                        fullname: au.fullname,
                        uniqueId: au.uniqueId,
                        id: au.id,
                    };
                }),
                organizationUnitName: row.userOrganizationUnitName,
                canListerApprove: row.canListerApprove,
                isHR: row.isHR,
                order: row.order,
                status: row.status,
                approvementDate: row.approvementDate,
                note: row.note,
            };
        }),
    });
});
module.exports.listPPermissionApprovementsForUserForHR = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let dbResult = await dal_manager_1.dbManager.accessPacs.listPPermissionUserApprovementInfo(organizationId, ppermissionId, userId, null);
    res.json({
        overallStatus: dbResult.overallStatus,
        items: dbResult.items.map((row) => {
            return {
                approverUsers: row.approverUsers.map((au) => {
                    return {
                        userCaptions: au.captionLines,
                        fullname: au.fullname,
                        uniqueId: au.uniqueId,
                        id: au.id,
                    };
                }),
                organizationUnitName: row.userOrganizationUnitName,
                canListerApprove: row.canListerApprove,
                isHR: row.isHR,
                order: row.order,
                status: row.status,
                approvementDate: row.approvementDate,
                note: row.note,
            };
        }),
    });
});
module.exports.approvePPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForPPermissionWrite)(req, organizationId);
    const ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    const request = new api_pacs_v1_models_1.RestPacsApiV1.ApprovePPermissionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessPacs.approvePPermission(organizationId, ppermissionId, {
        note: request.note,
        status: request.status,
        approveUserId: authResult.userId,
        isHR: false,
    });
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "Could not approve ppermission" }));
    }
    const ppermissionStatus = await dal_manager_1.dbManager.accessPacs.getPPermissionStatusInfoWithMyApproval(organizationId, ppermissionId, authResult.userId, false);
    res.json(ppermissionStatus);
});
module.exports.approvePPermissionFromHR = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let request = new api_pacs_v1_models_1.RestPacsApiV1.ApprovePPermissionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.accessPacs.approvePPermission(organizationId, ppermissionId, {
        note: request.note,
        status: request.status,
        approveUserId: authResult.userId,
        isHR: true,
    });
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "Could not approve ppermission" }));
    }
    let ppermissionStatus = await dal_manager_1.dbManager.accessPacs.getPPermissionStatusInfoWithMyApproval(organizationId, ppermissionId, null, true);
    res.json(ppermissionStatus);
});
module.exports.approvePPermissionForUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionWrite)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let request = new api_pacs_v1_models_1.RestPacsApiV1.ApprovePPermissionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const ppermissionStatus = await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
        let dbResult = await dal_manager_1.dbManager.accessPacs.approvePPermissionForUser(organizationId, ppermissionId, userId, {
            note: request.note,
            status: request.status,
            approveUserId: authResult.userId,
            isHR: false,
        });
        if (!dbResult) {
            next((0, api_error_1.generateNotFoundApiError)({ message: "Could not approve ppermission" }));
        }
        return dal_manager_1.dbManager.accessPacs.getPPermissionStatusInfoWithMyApproval(organizationId, ppermissionId, authResult.userId, false);
    });
    (0, business_pacs_ppermission_1.sendNextPPermissionNotification)({ organizationId, ppermissionId, userIds: [userId], requesterUserId: authResult.userId });
    res.json(ppermissionStatus);
});
module.exports.approvePPermissionForUserFromHR = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    const organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    const authResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    const ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    const userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const request = new api_pacs_v1_models_1.RestPacsApiV1.ApprovePPermissionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.accessPacs.approvePPermissionForUser(organizationId, ppermissionId, userId, {
        note: request.note,
        status: request.status,
        approveUserId: authResult.userId,
        isHR: true,
    });
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "Could not approve ppermission" }));
    }
    (0, business_pacs_ppermission_1.sendNextPPermissionNotification)({ organizationId, ppermissionId, userIds: [userId], requesterUserId: authResult.userId });
    const ppermissionStatus = await dal_manager_1.dbManager.accessPacs.getPPermissionStatusInfoWithMyApproval(organizationId, ppermissionId, null, true);
    res.json(ppermissionStatus);
});
module.exports.listSelfPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let request = new api_pacs_v1_models_1.RestPacsApiV1.SelfPPermissionFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.systemTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.listLeaves(trx, organizationId, jwt.userId, app_enums_1.enums.PPermissionInsertRole.Self, {
            userIds: [],
            ppermissionTypeId: request.ppermissionTypeId,
            startDateTime: request.startDateTime,
            endDateTime: request.endDateTime,
            take: request.pagination.take,
            skip: request.pagination.skip,
            status: request.status,
            ppermissionIds: request.ppermissionIds,
        });
    });
    res.json(result);
});
module.exports.listPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionRead)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.systemTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.listLeaves(trx, organizationId, authResult.userId, app_enums_1.enums.PPermissionInsertRole.Manager, {
            ppermissionTypeId: request.ppermissionTypeId,
            startDateTime: request.startDateTime,
            endDateTime: request.endDateTime,
            take: request.pagination.take,
            skip: request.pagination.skip,
            status: request.status,
            userIds: request.userIds,
            organizationUnitIds: request.organizationUnitIds,
            organizationUnitHierarchically: request.organizationUnitHierarchically,
            userGroupIds: request.userGroupIds,
            ppermissionIds: request.ppermissionIds,
            sortOrder: request.sortOrder,
            sortType: request.sortType,
        });
    });
    res.json(result);
});
module.exports.listPPermissionAll = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionReadAll)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.systemTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.listLeaves(trx, organizationId, authResult.userId, app_enums_1.enums.PPermissionInsertRole.HR, {
            ppermissionTypeId: request.ppermissionTypeId,
            startDateTime: request.startDateTime,
            endDateTime: request.endDateTime,
            take: request.pagination.take,
            skip: request.pagination.skip,
            status: request.status,
            userIds: request.userIds,
            ppermissionIds: request.ppermissionIds,
            sortOrder: request.sortOrder,
            sortType: request.sortType,
        });
    });
    res.json(result);
});
module.exports.listPPermissionHR = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let hrApprovementAuthResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let result = await dal_manager_1.dbManager.systemTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.listLeaves(trx, organizationId, hrApprovementAuthResult.userId, app_enums_1.enums.PPermissionInsertRole.HR, {
            ppermissionTypeId: request.ppermissionTypeId,
            startDateTime: request.startDateTime,
            endDateTime: request.endDateTime,
            take: request.pagination.take,
            skip: request.pagination.skip,
            status: request.status,
            userIds: request.userIds,
            organizationUnitIds: request.organizationUnitIds,
            organizationUnitHierarchically: request.organizationUnitHierarchically,
            userGroupIds: request.userGroupIds,
            ppermissionIds: request.ppermissionIds,
            sortOrder: request.sortOrder,
            sortType: request.sortType,
        });
    });
    res.json(result);
});
module.exports.listPPermissionHRForPdf = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let hrApprovementAuthResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.PermissionManagement,
                format: app_enums_1.enums.ReportFormat.Pdf,
                filter: options,
                organizationId: organizationId,
                requesterUserId: hrApprovementAuthResult.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report Permission Management is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.listPPermissionHRForExcel = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let hrApprovementAuthResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.PPermissionFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    try {
        let reportResult = await new Promise((resolve, reject) => {
            delete request.pagination;
            let options = request;
            messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendInstantReportRequest({
                code: app_enums_1.enums.ReportCode.PermissionManagement,
                format: app_enums_1.enums.ReportFormat.Excel,
                filter: options,
                organizationId: organizationId,
                requesterUserId: hrApprovementAuthResult.userId,
            }, app_config_1.appConfig.reportPdfExcelTimeoutInSeconds * 1000, (error, result) => {
                if (error) {
                    return reject(error);
                }
                resolve(result);
            });
        });
        app_logs_1.logger.info("Report Permission Management is ready in " + reportResult.duration + "ms");
        if (fs_1.default.existsSync(reportResult.filePath)) {
            res.download(reportResult.filePath, (err) => {
                if (err) {
                    throw err;
                }
                fs_1.default.unlinkSync(reportResult.filePath);
            });
        }
        else {
            throw new Error("Timeout");
        }
    }
    catch (error) {
        app_logs_1.logger.error(error.message || error);
        (0, report_util_1.throwDbReportError)(error.message, res);
    }
});
module.exports.getPPermissionHistoryOfUser = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionRead)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    let result = {
        items: [],
    };
    let historyList = await dal_manager_1.dbManager.accessLogPacs.getHistoryOfPPermission(organizationId, ppermissionId);
    historyList = (0, business_pacs_1.getHistoryOfPPermissionOfUser)(historyList, userId);
    result.items = await (0, business_pacs_1.transformPPermissionHistoryItemsToLocalizedMessages)(organizationId, historyList, locale);
    res.json(result);
});
module.exports.getPPermissionHistory = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionRead)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    let result = {
        items: [],
    };
    let historyList = await dal_manager_1.dbManager.accessLogPacs.getHistoryOfPPermission(organizationId, ppermissionId);
    result.items = await (0, business_pacs_1.transformPPermissionHistoryItemsToLocalizedMessages)(organizationId, historyList, locale);
    res.json(result);
});
module.exports.getSelfPPermissionHistory = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let jwt = req.auth_token;
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: jwt.userId });
    let result = {
        items: [],
    };
    let historyList = await dal_manager_1.dbManager.accessLogPacs.getHistoryOfPPermission(organizationId, ppermissionId);
    historyList = (0, business_pacs_1.getHistoryOfPPermissionOfUser)(historyList, jwt.userId);
    result.items = await (0, business_pacs_1.transformPPermissionHistoryItemsToLocalizedMessages)(organizationId, historyList, locale);
    res.json(result);
});
module.exports.getPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let isHR = undefined;
    try {
        isHR = (0, api_securityhelper_1.hasHRApprovement)(req, organizationId);
    }
    catch (error) {
        isHR = {
            isHR: false,
            isUnitManager: false,
            permittedUnitIds: [],
        };
    }
    let requesterUserId = (0, api_securityhelper_1.getUserIdFromToken)(req);
    let role = isHR.isHR ? app_enums_1.enums.PPermissionInsertRole.HR : isHR.isUnitManager ? app_enums_1.enums.PPermissionInsertRole.Manager : app_enums_1.enums.PPermissionInsertRole.Self;
    let result = await dal_manager_1.dbManager.systemTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.listLeaves(trx, organizationId, requesterUserId, role, {
            take: 1,
            skip: 0,
            userIds: [],
            ppermissionIds: [ppermissionId],
        });
    });
    if (result.items.length === 0) {
        (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("PACS.PPERMISSION_GET_AUTHORIZATION_ERROR");
    }
    else {
        res.json(result.items[0]);
    }
});
module.exports.removePPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let approvementResult = (0, api_securityhelper_1.hasHRApprovement)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    const dbResult = await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
        return dal_manager_1.dbManager.accessPacs.removePPermission(organizationId, ppermissionId, approvementResult.isHR, jwt.userId, approvementResult.isUnitManager, approvementResult.permittedUnitIds);
    });
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "Could not remove ppermission" }));
    }
    res.json();
});
module.exports.uploadAttachment = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let fileData = req.swagger.params.upfile.value;
    let fileName = req.swagger.params.fileName.value;
    const validExtensions = ["png", "pdf", "jpg", "jpeg", "doc", "docx"];
    const fileNameParts = fileData.originalname.split(".");
    if (!fileNameParts.length || !fileNameParts[fileNameParts.length - 1] || !validExtensions.includes(fileNameParts[fileNameParts.length - 1])) {
        res.status(400).json({ message: "Uploaded file is not valid!" });
        return;
    }
    const fileTypeResult = await file_type_1.default.fromBuffer(fileData.buffer);
    if (fileTypeResult.mime !== fileData.mimetype) {
        app_logs_1.logger.error(`Mime Type mismatch during file upload @ uploadAttachment, request mimetype: ${fileData.mimetype} content mimetype: ${fileTypeResult.mime}`);
        res.status(400).json({ message: "Uploaded file is not valid!" });
        return;
    }
    else {
        let dbResult = await dal_manager_1.dbManager.accessPacs.uploadPPermissionAttachment(organizationId, fileName);
        if (!dbResult) {
            next((0, api_error_1.generateNotFoundApiError)({ message: "Could not add attachment" }));
        }
        app_filesystem_1.fileSystem.addFile(organizationId, dbResult, fileData);
        res.json(dbResult);
    }
});
module.exports.getAttachmentOfPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let attachmentId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "attachmentId");
    let fileName = await dal_manager_1.dbManager.accessPacs.getAttachmentOfPermission(organizationId, attachmentId);
    if (fileName) {
        res.writeHead(200, {
            "Content-Type": "application/octet-stream",
            "Content-Disposition": "attachment; filename=" + fileName,
        });
        let stream = await app_filesystem_1.fileSystem.getReadStream(organizationId, attachmentId);
        if (stream) {
            stream.pipe(res);
        }
        else {
            (0, dal_access_error_1.throwDbAccessNotFoundError)(`Related file not found at file system!`);
        }
    }
    else {
        res.json();
    }
});
module.exports.removeAttachmentOfPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let attachmentId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "attachmentId");
    let dbResult = await dal_manager_1.dbManager.accessPacs.removePPermissionAttachment(organizationId, attachmentId);
    if (!dbResult) {
        next((0, api_error_1.generateNotFoundApiError)({ message: "Could not remove attachment of ppermission" }));
    }
    res.json();
});
module.exports.getSelfPPermissionInfoAndClaims = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let year = req.swagger.params.year.value;
    let month = req.swagger.params.month.value;
    let stamp = new Date();
    stamp.setFullYear(year, month - 1);
    try {
        let usages = await (0, business_pacs_claim_1.getPPermissionClaimAndUsage)(organizationId, jwt.userId, new Date());
        let infoList = [];
        for (const p of usages) {
            if (p.type.showRemainingCount) {
                if (p.type.isDailyScheduled) {
                    infoList.push({
                        ppermissionType: p.type,
                        used: p.usage,
                        left: p.left,
                        total: p.claim,
                        dueDate: p.dueDate,
                    });
                }
                else {
                    infoList.push({
                        ppermissionType: p.type,
                        used: +(p.usage / 60).toFixed(2),
                        left: +(p.left / 60).toFixed(2),
                        total: +(p.claim / 60).toFixed(2),
                        dueDate: p.dueDate,
                    });
                }
            }
        }
        res.json({
            ppermissionClaimInfoList: infoList,
        });
    }
    catch (err) {
        app_logs_1.logger.error(err);
        if (err.message && err.message === "ERRORS.PACS.EMPLOYEESTARTDATEMISSING") {
            (0, dal_access_error_1.throwDbAccessNotFoundErrorTr)("ERRORS.PACS.EMPLOYEESTARTDATEMISSING");
        }
        else {
            res.json(err);
        }
    }
});
module.exports.validatePPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionRead)(req, organizationId);
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: authResult.userId });
    let request = req.body;
    let validator = new business_pacs_validation_1.ValidateUserPPermission();
    let dbResult = await validator.validatePPermission(organizationId, request.ppermissionTypeId, request.userIds, locale, new Date(request.dateRange.startDateTime), new Date(request.dateRange.endDateTime));
    res.json(dbResult);
});
module.exports.validateSelfPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let request = req.body;
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: jwt.userId });
    let validator = new business_pacs_validation_1.ValidateUserPPermission();
    let dbResult = await validator.validateUserPPermission(organizationId, request.ppermissionTypeId, jwt.userId, locale, new Date(request.dateRange.startDateTime), new Date(request.dateRange.endDateTime), request.ppermissionId);
    res.json(dbResult);
});
module.exports.listPPermissionUsers = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionRead)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let request = new api_pacs_v1_models_1.RestPacsApiV1.UserListWithPagination(req.body, new api_validatorhelper_1.ValidatorHelper());
    const dbResult = await dal_manager_1.dbManager.organizationTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.listLeaves(trx, organizationId, authResult.userId, app_enums_1.enums.PPermissionInsertRole.Manager, {
            userIds: request.userIds,
            take: request.take,
            skip: request.skip,
            ppermissionIds: [ppermissionId],
        });
    }, authResult.userId, organizationId);
    res.json((0, business_pacs_ppermission_1.parsePPermissionListToUserInfoList)(dbResult));
});
module.exports.listPPermissionUsersForHR = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionHRApprovement)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let request = new api_pacs_v1_models_1.RestPacsApiV1.UserListWithPagination(req.body, new api_validatorhelper_1.ValidatorHelper());
    let dbResult = await dal_manager_1.dbManager.organizationTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.listLeaves(trx, organizationId, authResult.userId, app_enums_1.enums.PPermissionInsertRole.HR, {
            userIds: request.userIds,
            organizationUnitIds: request.organizationUnitIds,
            organizationUnitHierarchically: request.organizationUnitHierarchically,
            userGroupIds: request.userGroupIds,
            take: request.take,
            skip: request.skip,
            ppermissionIds: [ppermissionId],
        });
    }, authResult.userId, organizationId);
    res.json((0, business_pacs_ppermission_1.parsePPermissionListToUserInfoList)(dbResult));
});
module.exports.upsertPPermissionUserSignStatus = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForPPermissionRead)(req, organizationId);
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let dbResult = await dal_manager_1.dbManager.accessPacs.upsertPPermissionUserSignStatus(organizationId, ppermissionId, userId, req.body.status);
    res.json();
});
module.exports.obtainUserSelectionSessionForPPermission = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let ppermissionId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.id.value, "ppermissionId");
    let force = req.swagger.params.force.value;
    let authResult = (0, api_securityhelper_1.authorizeForObtainUserSelectionSessionForGeneric)(req, organizationId);
    let dbResult = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        return dal_manager_1.dbManager.accessUser.obtainUserSelectionSession(organizationId, {
            requestUserId: authResult.jwt.userId,
            type: app_enums_1.enums.UserSelectionSessionType.PPermission,
            relatedItemId: ppermissionId,
            forceToCreate: force,
            isAuthorizedForAllUsers: authResult.hasOrganizationWideWritePPermission && authResult.hasOrganizationWideReadBasic,
            authorizedOrganizationUnitIds: authResult.readBasicOrganizationUnitIds ?? null,
        }, trx);
    }, authResult.jwt.userId, organizationId);
    if (!dbResult.upserted) {
        throw (0, api_error_1.generateConflictError)({
            message: "There is already a session for that ppermission. Please see detailes.",
            details: dbResult.overwritten,
        });
    }
    let result = {
        sessionId: dbResult.sessionId,
        expirationUtc: dbResult.expirationUtc,
        justRefreshed: dbResult.justRefreshed,
        overwritten: dbResult.overwritten
            ? {
                ownerUserId: dbResult.overwritten.ownerUserId,
                sessionId: dbResult.overwritten.sessionId,
                expirationUtc: dbResult.overwritten.expirationUtc,
                captionLines: dbResult.overwritten.captionLines,
            }
            : undefined,
    };
    res.json(result);
});
module.exports.requestPPermissionDeputyInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let jwt = req.auth_token;
    let request = req.body;
    let response = null;
    let hook = business_hooks_1.armonHookManager.getRequestDeputyInfoHook(organizationId);
    if (hook) {
        response = hook({
            organizationId: organizationId,
            userIds: [jwt.userId],
            startDateTime: request.dateRange.startDateTime,
            endDateTime: request.dateRange.endDateTime,
            ppermissionTypeId: request.ppermissionTypeId,
        });
    }
    res.json(response);
});
module.exports.getCurrentPPermissionInfo = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendanceControlReport)(req, organizationId);
    let isHR = (0, api_securityhelper_1.hasHRRead)(req, organizationId);
    let result = await dal_manager_1.dbManager.accessPacs2.getCurrentPPermissionOverview(organizationId, authResult.userId, isHR ? dal_constants_1.DalConstants.PPermissionInsertRole.HR : dal_constants_1.DalConstants.PPermissionInsertRole.Manager);
    res.json(result);
});
module.exports.listCurrentPPermissionInfoUsers = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let authResult = (0, api_securityhelper_1.authorizeForAttendanceControlReport)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.CurrentWorkingSummaryPPermissionsFilter(req.body, new api_validatorhelper_1.ValidatorHelper());
    let isHR = (0, api_securityhelper_1.hasHRRead)(req, organizationId);
    let result = await dal_manager_1.dbManager.accessPacs2.listCurrentPPermissionOverview(organizationId, authResult.userId, isHR ? dal_constants_1.DalConstants.PPermissionInsertRole.HR : dal_constants_1.DalConstants.PPermissionInsertRole.Manager, request.pagination, request.currentWorkingSummaryPPermissionType, request.organizationUnitIds);
    res.json(result);
});
module.exports.editEmployeeAnnualPPermissionManually = (0, api_util_1.globalRouteHandler)(async (req, res, next) => {
    let organizationId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.oid.value, "organizationId");
    let userId = (0, api_validatorhelper_1.validateUUID)(req.swagger.params.uid.value, "userId");
    let authResult = (0, api_securityhelper_1.authorizeForManualPPermissionWrite)(req, organizationId);
    let request = new api_pacs_v1_models_1.RestPacsApiV1.ManualAnnualPPermissionRequest(req.body, new api_validatorhelper_1.ValidatorHelper());
    let jwt = req.auth_token;
    if (authResult.permittedUnitIds) {
        let orgUnitsOfUser = await dal_manager_1.dbManager.accessOrganizationUnit.listOrganizationUnitsOfUser(organizationId, userId);
        if (!orgUnitsOfUser.some((orgUnitId) => authResult.permittedUnitIds.includes(orgUnitId))) {
            const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId: jwt.userId });
            (0, dal_access_error_1.throwDbAccessAuthorizationError)(i18n_1.default.__({ phrase: "PACS.PPERMISSION_ANNUAL_MANUAL_EDIT_AUTHORIZATION_ERROR", locale: locale }));
        }
    }
    await dal_manager_1.dbManager.accessPacs2.updateRemainingAnnualPPermissionOfUser(organizationId, userId, request.remainingAnnualPPermission);
    res.json();
});
