"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiArmonV10Controller = void 0;
const express_1 = __importDefault(require("express"));
const fs_1 = __importDefault(require("fs"));
const app_logs_1 = require("../../../app.logs");
const business_device_1 = require("../../../business/business.device");
const dal_manager_1 = require("../../../dal/dal.manager");
const messagebroker_models_1 = require("../../../messageBroker/messagebroker.models");
const messageBroker_server_to_device_pub_1 = require("../../../messageBroker/messageBroker.server-to-device.pub");
const api_auth_1 = require("../../api.auth");
const api_error_1 = require("../../api.error");
const api_models_common_1 = require("../../api.models.common");
const api_util_1 = require("../../api.util");
const api_validatorhelper_1 = require("../../api.validatorhelper");
const api_armonv10_model_1 = require("./api.armonv10.model");
class ApiArmonV10Controller {
    constructor(app) {
        this.upsertFingerPrintCredential = async (req, res) => {
            let validator = new api_validatorhelper_1.ValidatorHelper();
            let request = new api_armonv10_model_1.ArmonV10ApiModelsV1.FingerPrintRegistryItem(req.body, validator);
            validator.finalize();
            await dal_manager_1.dbManager.accessUser.upsertFingerPrintCredentialFromDevice(req.organizationId, req.deviceId, request);
            res.json();
        };
        this.removeFingerPrintCredential = async (req, res) => {
            let validator = new api_validatorhelper_1.ValidatorHelper();
            let request = new api_armonv10_model_1.ArmonV10ApiModelsV1.RemoveFingerPrint(req.body, validator);
            validator.finalize();
            await dal_manager_1.dbManager.accessUser.removeFingerPrintCredentialFromDevice(req.organizationId, req.deviceId, request);
            res.json();
        };
        this.askDeviceData = async (req, res) => {
            let validator = new api_validatorhelper_1.ValidatorHelper();
            let request = new api_models_common_1.ApiModelsCommon.AskDeviceDataZip(req.body, validator);
            validator.finalize();
            let dbResult = await dal_manager_1.dbManager.accessDevice.getDataWithUsersForArmonOneSeries(req.deviceId, req.organizationId, request.configurationAndSettings, request.users, request.credentialTypes, request.currentSoftwareVersion);
            let result = {
                systemDateISO: new Date().toISOString(),
                configurationAndSettings: dbResult.configurationAndSettings,
                users: dbResult.users,
                lastSyncISO: dbResult.lastSyncISO,
            };
            (0, api_auth_1.sendZippedJsonResponse)(result, res);
        };
        this.askChanges = async (req, res) => {
            let sessionId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "sessionId");
            let items = await dal_manager_1.dbManager.accessDevice.getWholeChanges(req.deviceId, sessionId, req.organizationId);
            (0, api_auth_1.sendZippedJsonResponse)(items, res);
        };
        this.askChangeSessionCompleted = async (req, res) => {
            let sessionId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "sessionId");
            let remainingItemsCount = await dal_manager_1.dbManager.accessDevice.completeGetChangesSession(req.deviceId, sessionId, req.organizationId);
            res.json({
                remainingItemsCount: remainingItemsCount,
                systemDateISO: new Date().toISOString(),
            });
        };
        this.uploadOfflineLogs = async (req, res) => {
            let request = req.body;
            app_logs_1.logger.info("[ADAPTER] offline logs for deviceId:" + req.deviceId);
            let d1 = new Date();
            await dal_manager_1.dbManager.accessLog.checkUploadLogIdExits(req.organizationId, request.uploadId);
            let count = 0;
            let organizationId = req.organizationId;
            if (request.accessLogs && request.accessLogs.length > 0) {
                await dal_manager_1.dbManager.accessDevice.uploadDeviceAccessLogs(organizationId, request.accessLogs);
                count += request.accessLogs.length;
            }
            if (request.statusSensorLogs && request.statusSensorLogs.length > 0) {
                await dal_manager_1.dbManager.accessLog.uploadStatusSensorLogs(organizationId, request.statusSensorLogs);
                count += request.statusSensorLogs.length;
            }
            if (request.tamperSwitchLogs && request.tamperSwitchLogs.length > 0) {
                await dal_manager_1.dbManager.accessLog.uploadTamperSwitchLogs(organizationId, request.tamperSwitchLogs);
                count += request.tamperSwitchLogs.length;
            }
            if (request.counterLogs && request.counterLogs.length > 0) {
                await dal_manager_1.dbManager.accessLog.uploadCounterLogs({ organizationId, logs: request.counterLogs });
                count += request.counterLogs.length;
            }
            if (request.systemStatusLogs && request.systemStatusLogs.length > 0) {
                await (0, business_device_1.uploadDeviceSystemStatusLogs)(organizationId, req.deviceId, request.systemStatusLogs);
                count += request.systemStatusLogs.length;
            }
            if (request.fingerPrintRegisteryLogs && request.fingerPrintRegisteryLogs.length > 0) {
                await dal_manager_1.dbManager.accessUser.upsertFingerPrintCredentialsFromDevice(organizationId, req.deviceId, request.fingerPrintRegisteryLogs);
            }
            if (request.fingerPrintRemoveLogs && request.fingerPrintRemoveLogs.length > 0) {
                await dal_manager_1.dbManager.accessUser.removeFingerPrintCredentialsFromDevice(organizationId, req.deviceId, request.fingerPrintRegisteryLogs);
            }
            await dal_manager_1.dbManager.accessLog.addUploadLogId(organizationId, req.deviceId, request.uploadId, count);
            res.json({
                uploadId: request.uploadId,
            });
            let deviceIds = await dal_manager_1.dbManager.accessRegion.getOrganizationRegionRelatedDeviceIds(organizationId);
            for (let deviceId of deviceIds) {
                messageBroker_server_to_device_pub_1.amqpServerToDevicePub.sendToExchange(deviceId, {
                    e: messagebroker_models_1.ServerToDeviceMessageType.GetChangesRequired,
                });
            }
            let d2 = new Date();
            app_logs_1.logger.info("[ADAPTER] offline logs for:" + req.deviceId + ":" + (d2.getTime() - d1.getTime()));
        };
        this.failSnapshot = async (req, res) => {
            let snapshotId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "snapshotId");
            let dbResult = await dal_manager_1.dbManager.accessLog.uploadSnapshot(req.organizationId, req.deviceId, snapshotId, false, null);
            if (!dbResult)
                throw (0, api_error_1.generateNotFoundApiError)({ message: "Unable to fail snapshot" });
            res.json();
        };
        this.uploadSnapshot = async (req, res) => {
            let snapshotId = (0, api_validatorhelper_1.validateUUID)(req.params.sid, "snapshotId");
            let snapshot = req.body.snapshot.data;
            let actionUtc = req.body.actionUtc;
            let dbResult = await dal_manager_1.dbManager.accessLog.uploadSnapshot(req.organizationId, req.deviceId, snapshotId, true, actionUtc, snapshot);
            if (!dbResult)
                throw (0, api_error_1.generateNotFoundApiError)({ message: "Unable to upload snapshot" });
            res.json();
        };
        this.listUserGroups = async (req, res) => {
            let dbResult = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsForControlPanel(req.deviceId);
            res.json(dbResult);
        };
        this.listAccessRuleHistory = async (req, res) => {
            let dbResult = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                return dal_manager_1.dbManager.accessLog.listAccessRuleHistory(req.organizationId, trx);
            });
            res.json(dbResult);
        };
        this.listUserRegionTicketUnitsInfo = async (req, res) => {
            let dbResult = await dal_manager_1.dbManager.accessSocial.listUserRegionTicketsForControlPanel(req.deviceId);
            res.json(dbResult);
        };
        this.downloadUpdateFile = async (req, res) => {
            let deviceBasic = await dal_manager_1.dbManager.accessDevice.getDeviceBasic(req.organizationId, req.deviceId);
            let updateFile = await dal_manager_1.dbManager.accessDevice.getUpdateFile(req.organizationId, deviceBasic.brand, deviceBasic.model);
            if (updateFile) {
                if (fs_1.default.existsSync(updateFile.updateFilePath)) {
                    res.download(updateFile.updateFilePath);
                }
                else {
                    res.status(400).send("File does not exists in the specified location");
                }
            }
            else {
                res.status(404).send();
            }
        };
        this.getServerDetailes = (req, res) => {
            res.json({
                version: "0.9.1",
                message: "armon1 end point is working!",
            });
        };
        this._router = express_1.default.Router();
        this._router.get("/", this.getServerDetailes);
        this._router.post("/askdevicedata", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.askDeviceData));
        this._router.get("/askchanges/:sid", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.askChanges));
        this._router.get("/askchanges/:sid/completed", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.askChangeSessionCompleted));
        this._router.post("/uploadofflinelogs", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.uploadOfflineLogs));
        this._router.post("/snapshot/:sid", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.uploadSnapshot));
        this._router.get("/snapshot/:sid/fail", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.failSnapshot));
        this._router.get("/usergroups", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.listUserGroups));
        this._router.get("/accessrulehistory", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.listAccessRuleHistory));
        this._router.get("/userregionticketunits", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.listUserRegionTicketUnitsInfo));
        this._router.post("/user/addfingerprintcredential", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.upsertFingerPrintCredential));
        this._router.post("/user/removefingerprintcredential", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.removeFingerPrintCredential));
        this._router.get("/updatefile", api_auth_1.armonV10Authorize, (0, api_util_1.globalRouteHandler)(this.downloadUpdateFile));
        app.use("/av10", express_1.default.json({ limit: "100mb" }), this._router);
        app.use("/a1", express_1.default.json({ limit: "100mb" }), this._router);
    }
}
exports.ApiArmonV10Controller = ApiArmonV10Controller;
