"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidatorHelper = exports.validateIsBoolean = exports.validateThumbnail = exports.validateUUID = exports.isNullOrUndefined = void 0;
const moment_1 = __importDefault(require("moment"));
const app_enums_1 = require("../app.enums");
const api_error_1 = require("./api.error");
function isNullOrUndefined(value) {
    return value === undefined || value === null;
}
exports.isNullOrUndefined = isNullOrUndefined;
const uuidPattern = /^[0-9a-f]{8}-[0-9a-f]{4}-[1-4][0-9a-f]{3}-[0-9a-f]{4}-[0-9a-f]{12}$/i;
const regexHHmmss = new RegExp(/([0-1][0-9]|2[0-3]):([0-5][0-9]):([0-5][0-9])/);
function validateUUID(input, fieldName) {
    if (!uuidPattern.test(input)) {
        throw (0, api_error_1.generateBadRequestApiError)({
            message: "Request validation error",
            validationerrors: [
                {
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                    fields: [fieldName],
                    message: "Invalid UUIDV4.",
                    url: "https://tools.ietf.org/html/rfc4122",
                },
            ],
        });
    }
    return input;
}
exports.validateUUID = validateUUID;
function validateThumbnail(input, helper, finalize) {
    let data = input;
    if (input.includes("base64,")) {
        let prefix = input.split(",")[0];
        data = input.split(",")[1];
        helper
            .validateString({
            field: "prefix",
            optional: false,
            input: prefix,
        })
            .validateRegex("prefix", prefix, false, /(^data:image\/png;base64$)|(^data:image\/jpeg;base64$)/);
    }
    helper
        .validateString({
        field: "data",
        optional: false,
        input: data,
    })
        .validateRegex("data", data, false, /^(?:[A-Za-z0-9+\/]{4})*(?:[A-Za-z0-9+\/]{4}|[A-Za-z0-9+\/]{3}=|[A-Za-z0-9+\/]{2}={2})$/);
    if (finalize) {
        helper.finalize();
    }
}
exports.validateThumbnail = validateThumbnail;
function validateIsBoolean(input, fieldName) {
    if (input === undefined || input === null || (input !== true && input !== false && input !== "true" && input !== "false")) {
        throw (0, api_error_1.generateBadRequestApiError)({
            message: "Request validation error",
            validationerrors: [
                {
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                    fields: [fieldName],
                    message: "Invalid Boolean.",
                    url: "https://tools.ietf.org/html/rfc4122",
                },
            ],
        });
    }
    return input;
}
exports.validateIsBoolean = validateIsBoolean;
class ValidatorHelper {
    constructor() {
        this._errors = [];
    }
    get errors() {
        return this._errors;
    }
    finalize() {
        if (this._errors.length > 0) {
            throw (0, api_error_1.generateBadRequestApiError)({
                message: "Request validation error",
                validationerrors: this._errors,
            });
        }
    }
    mergeValidator(validator) {
        this._errors = this._errors.concat(validator.errors);
    }
    addFieldNotFoundError(field, message) {
        this._errors.push({
            code: app_enums_1.enums.ValidationErrorCode.NOT_FOUND,
            fields: [field],
            message: message || "Field is not found",
        });
        return this;
    }
    addFieldMustBeUniqueFieldError(field, message) {
        this._errors.push({
            code: app_enums_1.enums.ValidationErrorCode.MUST_BE_UNIQUE,
            fields: [field],
            message: message || "Field must be a unique field",
        });
        return this;
    }
    addArrayMustNotBeEmptyError(field, message) {
        this._errors.push({
            code: app_enums_1.enums.ValidationErrorCode.ARRAY_EMPTY,
            fields: [field],
            message: message || "Array must not be empty",
        });
        return this;
    }
    addContradictionError(fields, message) {
        this._errors.push({
            code: app_enums_1.enums.ValidationErrorCode.CONTRADICTION,
            fields: fields,
            message: message || "Contradiction between the fields",
        });
        return this;
    }
    addRangeError(fields, message) {
        this._errors.push({
            code: app_enums_1.enums.ValidationErrorCode.RANGE,
            fields: fields,
            message: "Invalid  Range. 'Start' value should be smaller then 'End' value",
        });
        return this;
    }
    validateCheckboxFormFieldValue(value, field) {
        let options = field.options;
        if (!field.required || options.type === app_enums_1.enums.CheckboxFormFieldType.Tristate) {
            if (!isNullOrUndefined(value) && value !== true && value !== false) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                    fields: [field.name],
                    message: "Invalid Tristate Checkbox Value. The value should be null, true or false",
                });
            }
        }
        else {
            if (value !== true && value !== false) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                    fields: [field.name],
                    message: "Invalid Checkbox Value. The value should be true or false",
                });
            }
        }
    }
    validateDateFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        let options = field.options;
        let parsedValue = Date.parse(value);
        if (isNaN(parsedValue)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field.name],
                message: "Invalid Date. Use ISO 8601 format.",
                url: "https://www.w3.org/TR/NOTE-datetime",
            });
        }
        else {
            if (options.max && parsedValue > options.max.getTime()) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.MAXIMUM,
                    fields: [field.name],
                    message: "The date should be before the max value",
                });
            }
            if (options.min && parsedValue < options.min.getTime()) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.MINIMUM,
                    fields: [field.name],
                    message: "The date should be after the min value",
                });
            }
        }
    }
    validateDateTimeFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        let options = field.options;
        let parsedValue = Date.parse(value);
        if (isNaN(parsedValue)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field.name],
                message: "Invalid Date. Use ISO 8601 format.",
                url: "https://www.w3.org/TR/NOTE-datetime",
            });
        }
        else {
            if (options.max && parsedValue > options.max.getTime()) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.MAXIMUM,
                    fields: [field.name],
                    message: "The date should be before the max value",
                });
            }
            if (options.min && parsedValue < options.min.getTime()) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.MINIMUM,
                    fields: [field.name],
                    message: "The date should be after the min value",
                });
            }
        }
    }
    validateDateRangeFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        let options = field.options;
        let parsedValueStart = Date.parse(value.startDate);
        if (isNaN(parsedValueStart)) {
            if (options.isStartRequired)
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                    fields: [field.name],
                    message: "Invalid Date. Use ISO 8601 format.",
                    url: "https://www.w3.org/TR/NOTE-datetime",
                });
        }
        let parsedValueEnd = Date.parse(value.endDate);
        if (isNaN(parsedValueEnd)) {
            if (options.isEndRequired)
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                    fields: [field.name],
                    message: "Invalid Date. Use ISO 8601 format.",
                    url: "https://www.w3.org/TR/NOTE-datetime",
                });
        }
    }
    validateImageFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
    }
    validateNumberFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        let options = field.options;
        let parsedValue = parseFloat(value);
        if (isNaN(parsedValue)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field.name],
                message: "Invalid number",
            });
        }
        else {
            if (options.max && parsedValue > options.max) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.MAXIMUM,
                    fields: [field.name],
                    message: "The number should be before the max value",
                });
            }
            if (options.min && parsedValue < options.min) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.MINIMUM,
                    fields: [field.name],
                    message: "The numbre should be after the min value",
                });
            }
        }
    }
    validateRadioGroupFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        let options = field.options;
        let found = false;
        for (const option of options.options) {
            if (option.value === value) {
                found = true;
                break;
            }
        }
        if (!found) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_VALUE,
                fields: [field.name],
                message: "The value is not in the options",
            });
        }
    }
    validateSelectFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        let options = field.options;
        if (options.type !== app_enums_1.enums.SelectFormFieldType.TypeaheadServerField) {
            let found = false;
            for (const option of options.options) {
                if (option.value === value) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_VALUE,
                    fields: [field.name],
                    message: "The value is not in the options",
                });
            }
        }
    }
    validateTextFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        let options = field.options;
        if (typeof value !== "string") {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field.name],
                message: "The value should be string",
            });
        }
        else {
            if (options.maxLength && value.length > options.maxLength) {
                this._errors.push(this.generateMaxLengthError(field.name, options.maxLength));
            }
            if (options.minLength && value.length < options.minLength) {
                this._errors.push(this.generateMinLengthError(field.name, options.minLength));
            }
        }
    }
    validateTextareaFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        let options = field.options;
        if (typeof value !== "string") {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field.name],
                message: "The value should be string",
            });
        }
        else {
            if (options.maxLength && value.length > options.maxLength) {
                this._errors.push(this.generateMaxLengthError(field.name, options.maxLength));
            }
            if (options.minLength && value.length < options.minLength) {
                this._errors.push(this.generateMinLengthError(field.name, options.minLength));
            }
        }
    }
    validateTimeFormFieldValue(value, field) {
        if (isNullOrUndefined(value)) {
            if (!field.required) {
                return;
            }
            else {
                this._errors.push(this.generateRequiredError(field.name));
            }
        }
        if (regexHHmmss.test(value)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field.name],
                message: "The value should be time in format HH:mm:ss",
            });
        }
    }
    validateFormFieldValue(value, field) {
        if (field.regex) {
            let regex = new RegExp(field.regex);
            if (!regex.test(value) && field.required) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.REGEX,
                    fields: [field.name],
                    message: "The value is not satifies the regex",
                });
                return this;
            }
        }
        switch (field.type) {
            case app_enums_1.enums.FormFieldType.Checkbox:
                this.validateCheckboxFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.Date:
                this.validateDateFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.DateTime:
                this.validateDateTimeFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.Number:
                this.validateNumberFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.RadioGroup:
                this.validateRadioGroupFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.Select:
                this.validateSelectFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.Text:
                this.validateTextFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.Textarea:
                this.validateTextareaFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.Time:
                this.validateTimeFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.DateRange:
                this.validateDateRangeFormFieldValue(value, field);
                break;
            case app_enums_1.enums.FormFieldType.ImageCapture:
                this.validateImageFormFieldValue(value, field);
                break;
            default:
                break;
        }
        return this;
    }
    validateEnum(field, optional, type, input) {
        if (optional && input == null) {
            return this;
        }
        if (input == null) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Number.isInteger(input) || !type[input]) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.ENUM_MISMATCH,
                fields: [field],
                message: "Invalid enum value",
            });
        }
        return this;
    }
    validateEnumString(field, optional, type, input) {
        if (optional && input == null) {
            return this;
        }
        if (input == null) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (typeof input !== "string" || !type[input]) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.ENUM_MISMATCH,
                fields: [field],
                message: "Invalid enum value",
            });
        }
        return this;
    }
    validateEnumArray(field, optional, type, input) {
        if (optional && (input == null || input.length === 0)) {
            return this;
        }
        if (input == null) {
            this._errors.push(this.generateRequiredError(field));
        }
        else {
            for (let index = 0; index < input.length; index++) {
                if (!Number.isInteger(input[index]) || !type[input[index]]) {
                    this._errors.push({
                        code: app_enums_1.enums.ValidationErrorCode.ENUM_MISMATCH,
                        fields: [field],
                        message: "Array contains invalid enum value",
                    });
                }
            }
        }
        return this;
    }
    validateEnumArrayOfArray(field, optional, type, input) {
        if (optional && (input == null || input.length === 0)) {
            return this;
        }
        if (input == null) {
            this._errors.push(this.generateRequiredError(field));
        }
        else {
            for (let index = 0; index < input.length; index++) {
                let innerItem = input[index];
                if (!innerItem || innerItem.length === 0) {
                    this._errors.push({
                        code: app_enums_1.enums.ValidationErrorCode.ARRAY_EMPTY,
                        fields: [field],
                        message: "Array contains empty value",
                    });
                    return this;
                }
                for (let indexInner = 0; indexInner < innerItem.length; index++) {
                    if (!Number.isInteger(innerItem[indexInner]) || !type[innerItem[indexInner]]) {
                        this._errors.push({
                            code: app_enums_1.enums.ValidationErrorCode.ENUM_MISMATCH,
                            fields: [field],
                            message: "Array contains invalid enum value",
                        });
                    }
                }
            }
        }
        return this;
    }
    validateDate(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (isNaN(Date.parse(input))) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field],
                message: "Invalid Date. Use ISO 8601 format.",
                url: "https://www.w3.org/TR/NOTE-datetime",
            });
        }
        return this;
    }
    validateDateWithFormat(field, optional, format, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!(0, moment_1.default)(input, format).isValid()) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field],
                message: "Invalid Date. Use ISO 8601 format.",
                url: "https://www.w3.org/TR/NOTE-datetime",
            });
        }
        return this;
    }
    validateIsBoolean(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (input === undefined || input === null) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (input !== true && input !== false && input !== "true" && input !== "false") {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field],
                message: "Invalid Boolean value. Use 'true' or 'false'",
            });
        }
        return this;
    }
    validateIsArrayOfDate(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Array.isArray(input)) {
            this._errors.push(this.generateNotArrayError(field));
        }
        else {
            for (let i = 0; i < input.length; i++) {
                if (isNaN(Date.parse(input[i]))) {
                    this._errors.push({
                        code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                        fields: [field],
                        message: "Invalid Date. Use ISO 8601 format.",
                        url: "https://www.w3.org/TR/NOTE-datetime",
                    });
                    break;
                }
            }
        }
        return this;
    }
    validateArrayLength(field, optional, minLength, maxLength, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Array.isArray(input)) {
            this._errors.push(this.generateNotArrayError(field));
        }
        else {
            let length = input.length;
            if (minLength > -1 && length < minLength) {
                this._errors.push(this.generateMinLengthError(field, minLength));
            }
            if (maxLength > -1 && length > maxLength) {
                this._errors.push(this.generateMaxLengthError(field, minLength));
            }
        }
        return this;
    }
    validateIsArrayOfBoolean(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Array.isArray(input)) {
            this._errors.push(this.generateNotArrayError(field));
        }
        else {
            for (let i = 0; i < input.length; i++) {
                if (input[i] !== true && input[i] !== false) {
                    this._errors.push({
                        code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                        fields: [field],
                        message: "Invalid Boolean.",
                        url: "",
                    });
                    break;
                }
            }
        }
        return this;
    }
    validateDateRange(fields, params) {
        if ((params.startOptional && !params.start) || (params.endOptional && !params.end)) {
            return this;
        }
        if (params.start && params.end && params.end <= params.start) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.RANGE,
                fields: fields,
                message: `Invalid Date Range. '${fields[0]}' value should be smaller then '${fields[1]}' value`,
            });
        }
        return this;
    }
    validateObjectInstance(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        return this;
    }
    validateObject(field, optional, input, validator) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        this.mergeValidator(validator);
        return this;
    }
    validateObjectSingleKey(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        if (Object.keys(input).length > 1) {
            this._errors.push(this.generateSingleKeyError(field));
        }
        return this;
    }
    validateObjectArray(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Array.isArray(input)) {
            this._errors.push(this.generateNotArrayError(field));
        }
        return this;
    }
    validateUUID(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (input && !uuidPattern.test(input)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field],
                message: "Invalid UUIDV4.",
                url: "https://tools.ietf.org/html/rfc4122",
            });
        }
        return this;
    }
    validateUUIDAndFinalize(field, optional, input) {
        if (optional && !input) {
            return null;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (input && !uuidPattern.test(input)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field],
                message: "Invalid UUIDV4.",
                url: "https://tools.ietf.org/html/rfc4122",
            });
        }
        this.finalize();
        return input;
    }
    validateUUIDsAndFinalize(params) {
        let result = {};
        for (let index = 0; index < params.length; index++) {
            let param = params[index];
            if (param.optional && !param.input) {
                result[param.field] = param.input;
            }
            if (!param.input) {
                this._errors.push(this.generateRequiredError(param.field));
            }
            else if (param.input && !uuidPattern.test(param.input)) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                    fields: [param.field],
                    message: "Invalid UUIDV4.",
                    url: "https://tools.ietf.org/html/rfc4122",
                });
            }
            else {
                result[param.field] = param.input;
            }
        }
        this.finalize();
        return result;
    }
    validateUUIDArray(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Array.isArray(input)) {
            this._errors.push(this.generateNotArrayError(field));
        }
        else {
            for (let i = 0; i < input.length; i++) {
                if (!uuidPattern.test(input[i])) {
                    this._errors.push({
                        code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                        fields: [field],
                        message: "Invalid UUIDV4.",
                        url: "https://tools.ietf.org/html/rfc4122",
                    });
                    break;
                }
            }
        }
        return this;
    }
    validateIsArrayOfInteger(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Array.isArray(input)) {
            this._errors.push(this.generateNotArrayError(field));
        }
        else {
            for (let i = 0; i < input.length; i++) {
                if (!Number.isInteger(input[i])) {
                    this._errors.push({
                        code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                        fields: [field],
                        message: "Contains not an integer value",
                    });
                    break;
                }
            }
        }
        return this;
    }
    validateInteger(params) {
        if (params.optional && !params.input) {
            return this;
        }
        if (params.input === null || params.input === undefined) {
            this._errors.push(this.generateRequiredError(params.field));
        }
        else if (!Number.isInteger(params.input)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [params.field],
                message: "Not an integer value",
            });
        }
        else {
            if (params.max && params.input > params.max) {
                this._errors.push(this.generateMaxValueError(params.field, params.max));
            }
            if (params.min && params.input < params.min) {
                this._errors.push(this.generateMinValueError(params.field, params.min));
            }
        }
        return this;
    }
    validateFloat(field, optional, input, min, max) {
        if (optional && !input) {
            return this;
        }
        let parsedValue = parseFloat(input);
        if (input === null || input === undefined) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (isNaN(parsedValue)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [field],
                message: "Not a float value",
            });
        }
        if (min !== null && min !== undefined) {
            if (input < min) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_VALUE,
                    fields: [field],
                    message: "Lower than minimum limit",
                });
            }
        }
        if (max !== null && max !== undefined) {
            if (input > max) {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_VALUE,
                    fields: [field],
                    message: "Greater than maximum limit",
                });
            }
        }
        return this;
    }
    validateIntegerRange(params) {
        let c = this._errors.length;
        this.validateInteger(params.smallNumberParams).validateInteger(params.bigNumberParams);
        if (c === this._errors.length) {
            if (!params.smallNumberParams.optional && !params.bigNumberParams.optional) {
                if (params.smallNumberParams.input >= params.bigNumberParams.input) {
                    this._errors.push({
                        code: app_enums_1.enums.ValidationErrorCode.RANGE,
                        fields: [params.smallNumberParams.field, params.bigNumberParams.field],
                        message: "Invalid number range.",
                    });
                }
            }
        }
        return this;
    }
    validateIsArrayOfString(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Array.isArray(input)) {
            this._errors.push(this.generateNotArrayError(field));
        }
        else {
            for (let i = 0; i < input.length; i++) {
                if (typeof input[i] !== "string") {
                    this._errors.push({
                        code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                        fields: [field],
                        message: "Contains not a string value",
                    });
                    break;
                }
            }
        }
        return this;
    }
    validateIsArrayOfArray(field, optional, input) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!Array.isArray(input)) {
            this._errors.push(this.generateNotArrayError(field));
        }
        else {
            for (let i = 0; i < input.length; i++) {
                if (!Array.isArray(input[i])) {
                    this._errors.push(this.generateNotArrayError(field + `[${i}]`));
                    break;
                }
            }
        }
        return this;
    }
    validateString(params) {
        if (params.optional && !params.input) {
            return this;
        }
        if (!params.input) {
            this._errors.push(this.generateRequiredError(params.field));
        }
        else if (typeof params.input !== "string") {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
                fields: [params.field],
                message: "The value should be string",
            });
        }
        else {
            if (params.maxLength && params.input.length > params.maxLength) {
                this._errors.push(this.generateMaxLengthError(params.field, params.maxLength));
            }
            if (params.minLength && params.input.length < params.minLength) {
                this._errors.push(this.generateMinLengthError(params.field, params.minLength));
            }
        }
        return this;
    }
    validateRegex(field, input, optional, regex) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        else if (!regex.test(input)) {
            this._errors.push({
                code: app_enums_1.enums.ValidationErrorCode.REGEX,
                fields: [field],
                message: "The value should be regex tested",
            });
        }
        return this;
    }
    validatePaginationRequest(field, input, optional, maxTake) {
        if (optional && !input) {
            return this;
        }
        if (!input) {
            this._errors.push(this.generateRequiredError(field));
        }
        if (input.take < 0 || (maxTake ? input.take > maxTake : input.take > 1000) || input.skip < 0 || isNaN(input.skip) || isNaN(input.take)) {
            if (maxTake ? input.take > maxTake : input.take > 1000) {
                this._errors.push(this.generateMaxValueError("take", maxTake ?? 1000));
            }
            else {
                this._errors.push({
                    code: app_enums_1.enums.ValidationErrorCode.INVALID_VALUE,
                    fields: [field],
                    message: "The value of pagination is incorrect",
                });
            }
        }
        return this;
    }
    generateNotArrayError(field) {
        return {
            code: app_enums_1.enums.ValidationErrorCode.INVALID_TYPE,
            fields: [field],
            message: "Waits for array",
        };
    }
    generateRequiredError(field) {
        return {
            code: app_enums_1.enums.ValidationErrorCode.REQUIRED,
            fields: [field],
            message: "Required field",
        };
    }
    generateMaxValueError(field, max) {
        return {
            code: app_enums_1.enums.ValidationErrorCode.MAXIMUM,
            fields: [field],
            message: "The value must be less than " + max,
        };
    }
    generateMinValueError(field, min) {
        return {
            code: app_enums_1.enums.ValidationErrorCode.MINIMUM,
            fields: [field],
            message: "The value must be more than " + min,
        };
    }
    generateMaxLengthError(field, maxLength) {
        return {
            code: app_enums_1.enums.ValidationErrorCode.MAX_LENGTH,
            fields: [field],
            message: "The value's length exceeds " + maxLength,
        };
    }
    generateMinLengthError(field, minLength) {
        return {
            code: app_enums_1.enums.ValidationErrorCode.MIN_LENGTH,
            fields: [field],
            message: "The value's length must be more than " + minLength,
        };
    }
    generateSingleKeyError(field) {
        return {
            code: app_enums_1.enums.ValidationErrorCode.NOT_SINGLE_KEY,
            fields: [field],
            message: "The value should contain only single key",
        };
    }
}
exports.ValidatorHelper = ValidatorHelper;
