"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatDateToArventoDateString = exports.synchronizeArventoAccessLogs = exports.isArventoIntegrationEnabled = exports.sendUserDisabledRequest = exports.driverQrScanRequest = exports.getArventoIntegrationConfigAndSecrets = void 0;
const axios_1 = __importDefault(require("axios"));
const dal_manager_1 = require("../db/../dal/dal.manager");
const dal_access_error_1 = require("../dal/access/dal.access.error");
const api_error_1 = require("./api.error");
const restapi_1 = require("../lib/es/models/restapi");
const dal_access_psql_organization_1 = require("../dal/access/psql/dal.access.psql.organization");
const dal_constants_1 = require("../dal/dal.constants");
const moment_1 = __importDefault(require("moment"));
const xml2js_1 = require("xml2js");
const app_logs_1 = require("../app.logs");
async function getArventoIntegrationConfigAndSecrets(organizationId, type, trx) {
    const integrations = await dal_manager_1.dbManager.accessOrganization.getOrganizationSettingsIntegrations({ organizationId, trx });
    if (!integrations) {
        return null;
    }
    const arventoIntegrationConfig = integrations.find((item) => item.type === type)?.config;
    if (!arventoIntegrationConfig?.secretId) {
        return null;
    }
    const arventoSecrets = await (0, dal_access_psql_organization_1.getOrganizationSecretById)(organizationId, arventoIntegrationConfig.secretId, trx);
    if (!arventoSecrets) {
        return null;
    }
    return {
        arventoIntegrationConfig,
        arventoSecrets,
    };
}
exports.getArventoIntegrationConfigAndSecrets = getArventoIntegrationConfigAndSecrets;
async function driverQrScanRequest(organizationId, userId, request, trx) {
    const uniqueId = await dal_manager_1.dbManager.accessUser.getUniqueIdFromUserId(organizationId, userId, trx);
    const arventoConfigSecrets = await getArventoIntegrationConfigAndSecrets(organizationId, restapi_1.IntegrationType.Arvento, trx);
    if (!arventoConfigSecrets) {
        (0, dal_access_error_1.throwDbAccessNotFoundError)("Arvento integration config or secrets not found!");
    }
    const arventoSecrets = arventoConfigSecrets.arventoSecrets;
    const arventoConfiguration = arventoConfigSecrets.arventoIntegrationConfig;
    const arventoUrl = new URL("driver-qr-scan", arventoSecrets?.baseUrl).toString();
    let response = null;
    try {
        response = await axios_1.default.post(arventoUrl, {
            person_id: uniqueId,
            qr_code_content: request.qrCodeContent,
            customer_id: arventoSecrets?.customerId,
            latitude: request.latitude,
            longitude: request.longitude,
            date_time: request.timestamp ?? new Date().toLocaleString(),
        }, {
            headers: {
                "Content-Type": "application/json",
                "X-API-KEY": arventoSecrets?.apiKey,
            },
            timeout: arventoConfiguration?.timeout ?? 5000,
        });
        if (arventoConfiguration?.accessLog && response.status >= 200 && response.status < 300) {
            let accessLogId = null;
            let accessLogErrors = [];
            const accessControlPoint = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(organizationId, dal_constants_1.DalConstants.defaultArventoDriverAcpId);
            if (!accessControlPoint) {
                accessLogErrors.push("Access control point not found");
                response.data.data = {
                    ...response.data.data,
                    accessLog: {
                        success: false,
                        accessLogId,
                        errors: accessLogErrors,
                    },
                };
                return response.data;
            }
            if ((0, moment_1.default)(request.timestamp ?? new Date()).isAfter(new Date())) {
                return accessLogErrors.push("invalid action date");
            }
            else {
                accessLogId = await dal_manager_1.dbManager.accessLog.addArventoDriverAccessLog(organizationId, {
                    accessControlPointId: accessControlPoint.id,
                    accessControlPointName: accessControlPoint.name,
                    userId: userId,
                    reason: dal_constants_1.DalConstants.libEnumsV2.AccessLogReason.Manual,
                    direction: dal_constants_1.DalConstants.AccessDirection.All,
                    actionUtc: new Date(request.timestamp ?? new Date()),
                    requesterUserId: userId,
                    isQrTriggered: false,
                    licencePlate: response.data?.data?.qr_match?.plate ?? "",
                }, trx);
            }
            if (!accessLogId) {
                accessLogErrors.push("Error while adding access log");
            }
            response.data.data = {
                ...response.data.data,
                accessLog: {
                    success: accessLogId ? true : false,
                    accessLogId,
                    errors: accessLogErrors,
                },
            };
        }
        return response.data;
    }
    catch (err) {
        throw (0, api_error_1.generateTranslatedError)(err.response?.status || 500, "ERRORS.CUSTOM.ARVENTO_SERVER_ERROR", {
            error_message: err.response?.data?.message + ": " + err.response?.data?.validation_errors?.join(" "),
        });
    }
}
exports.driverQrScanRequest = driverQrScanRequest;
async function sendUserDisabledRequest(uniqueId, organizationId) {
    const arventoConfigSecrets = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
        return await getArventoIntegrationConfigAndSecrets(organizationId, restapi_1.IntegrationType.Arvento, trx);
    });
    if (!arventoConfigSecrets) {
        app_logs_1.logger.error("Arvento integration config or secrets not found!");
        return;
    }
    const arventoSecrets = arventoConfigSecrets.arventoSecrets;
    const requestUrl = new URL("v1/vehicle.asmx/DeleteDriver", arventoSecrets?.vehicleServiceUrl);
    requestUrl.searchParams.append("Username", arventoSecrets?.vehicleServiceUsername);
    requestUrl.searchParams.append("PIN1", arventoSecrets?.vehicleServicePin1);
    requestUrl.searchParams.append("PIN2", arventoSecrets?.vehicleServicePin2);
    requestUrl.searchParams.append("Driver", "");
    requestUrl.searchParams.append("EmployeeNumber", uniqueId);
    requestUrl.searchParams.append("DeleteMappings", "true");
    try {
        let response = await axios_1.default.get(requestUrl.toString(), {
            timeout: 5000,
        });
        const parsedResponse = await (0, xml2js_1.parseStringPromise)(response.data);
        const responseString = parsedResponse?.["string"]?._ ?? "";
        app_logs_1.logger.info(`Successful driver disable request sent to Arvento with response: ${responseString}`);
    }
    catch (err) {
        app_logs_1.logger.error("Arvento Vehicle Server Error: " + { error_message: err.response?.data?.message + ": " + err.response?.data?.validation_errors?.join(" ") });
        return;
    }
}
exports.sendUserDisabledRequest = sendUserDisabledRequest;
async function isArventoIntegrationEnabled(organizationId) {
    const integrations = await dal_manager_1.dbManager.systemTransaction((trx) => dal_manager_1.dbManager.accessOrganization.getOrganizationSettingsIntegrations({ organizationId, trx }));
    return integrations?.some((item) => item.type === restapi_1.IntegrationType.Arvento) ?? false;
}
exports.isArventoIntegrationEnabled = isArventoIntegrationEnabled;
async function synchronizeArventoAccessLogs(organizationId, trx) {
    try {
        const accessLogs = await dal_manager_1.dbManager.accessLog.getArventoAccessLogsWithNodeId(organizationId, trx);
        if (!accessLogs || accessLogs.length === 0) {
            app_logs_1.logger.info("Arvento Region: There is no access logs to synchronize.");
            return true;
        }
        let startDate = accessLogs.reduce((min, log) => (log.actionUtc < min ? log.actionUtc : min), accessLogs[0].actionUtc);
        let endDate = accessLogs.reduce((max, log) => (log.actionUtc > max ? log.actionUtc : max), accessLogs[0].actionUtc);
        const arventoRegionData = await getRegionDataWithDate(organizationId, startDate, endDate, trx);
        if (!arventoRegionData) {
            app_logs_1.logger.error(`Arvento Region: No region data found for organization ${organizationId} between ${startDate} and ${endDate}`);
            return false;
        }
        for (const accessLog of accessLogs) {
            const currentRegion = findCurrentRegion(arventoRegionData, accessLog);
            if (currentRegion != null) {
                await dal_manager_1.dbManager.accessLog.updateArventoAccessLog(organizationId, accessLog.id, currentRegion, trx);
            }
        }
        return true;
    }
    catch (error) {
        app_logs_1.logger.error(`Error synchronizing Arvento access logs for organization ${organizationId}: ${error.message}`);
        return false;
    }
}
exports.synchronizeArventoAccessLogs = synchronizeArventoAccessLogs;
function findCurrentRegion(data, accessLog) {
    const { arventoNodeId, actionUtc } = accessLog;
    const deviceLogs = data[arventoNodeId];
    if (!deviceLogs || deviceLogs.length === 0) {
        return "";
    }
    deviceLogs.sort((a, b) => new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime());
    const activeRegions = new Map();
    const currentTime = new Date(actionUtc);
    let resultRegions = new Set();
    for (const log of deviceLogs) {
        const logTime = new Date(log.timestamp);
        if (isNaN(logTime.getTime())) {
            app_logs_1.logger.error(`Arvento Region: Invalid timestamp: ${log.timestamp}`);
            continue;
        }
        if (logTime > currentTime) {
            break;
        }
        if (log.status === "Bölgeye Giriş") {
            if (activeRegions.has(log.region)) {
                app_logs_1.logger.warn(`Arvento Region: Inconsistent data: consecutive entry record for ${log.region} (${log.timestamp})`);
            }
            activeRegions.set(log.region, log.timestamp);
        }
        else if (log.status === "Bölgeden Çıkış") {
            if (!activeRegions.has(log.region)) {
                app_logs_1.logger.warn(`Arvento Region: Inconsistent data: exit record for ${log.region} without entry record (${log.timestamp})`);
            }
            activeRegions.delete(log.region);
        }
    }
    for (const [region, entryTimestamp] of activeRegions) {
        if (new Date(entryTimestamp) <= currentTime) {
            resultRegions.add(region);
        }
    }
    return resultRegions.size > 0 ? Array.from(resultRegions).join(", ") : "";
}
async function getRegionDataWithDate(organizationId, startDate, endDate, trx) {
    if (!startDate || !endDate) {
        app_logs_1.logger.error("Arvento Region: Invalid startDate or endDate provided!");
        return null;
    }
    const arventoConfigSecrets = await getArventoIntegrationConfigAndSecrets(organizationId, restapi_1.IntegrationType.ArventoRegion, trx);
    if (!arventoConfigSecrets) {
        app_logs_1.logger.error("Arvento region integration config or secrets not found!");
        return null;
    }
    const arventoConfiguration = arventoConfigSecrets.arventoIntegrationConfig;
    const arventoSecrets = arventoConfigSecrets.arventoSecrets;
    const hoursBack = arventoConfiguration?.hoursBack || 4;
    startDate = new Date(startDate.getTime() - hoursBack * 60 * 60 * 1000);
    const requestUrl = new URL("v1/report.asmx/RegionAlarm", arventoSecrets?.regionServiceUrl);
    requestUrl.searchParams.append("Username", arventoSecrets?.regionServiceUsername);
    requestUrl.searchParams.append("PIN1", arventoSecrets?.regionServicePin1);
    requestUrl.searchParams.append("PIN2", arventoSecrets?.regionServicePin2);
    requestUrl.searchParams.append("StartDate", formatDateToArventoDateString(startDate));
    requestUrl.searchParams.append("EndDate", formatDateToArventoDateString(endDate));
    requestUrl.searchParams.append("Node", "");
    requestUrl.searchParams.append("Group", "");
    requestUrl.searchParams.append("Region", "");
    requestUrl.searchParams.append("Compress", "");
    requestUrl.searchParams.append("MinuteDif", "");
    requestUrl.searchParams.append("Language", "");
    try {
        let response = await axios_1.default.get(requestUrl.toString(), {
            timeout: 5000,
        });
        const xmlData = await (0, xml2js_1.parseStringPromise)(response.data);
        const entries = xmlData?.DataSet?.["diffgr:diffgram"]?.[0]?.NewDataSet?.[0]?.Table1 ?? [];
        let regionData = {};
        entries.forEach((entry) => {
            const deviceNo = entry.Cihaz_x0020_No[0];
            const region = entry.Coğrafi_x0020_Bölge[0];
            const status = entry.Durumu[0];
            const timestamp = new Date(entry.Tarih_x002F_Saat[0]);
            if (!regionData[deviceNo]) {
                regionData[deviceNo] = [];
            }
            regionData[deviceNo].push({
                region: region,
                status: status,
                timestamp: timestamp,
            });
        });
        return regionData;
    }
    catch (err) {
        app_logs_1.logger.error("Arvento Region Server Error: " + { error_message: err.response?.data?.message + ": " + err.response?.data?.validation_errors?.join(" ") });
        return null;
    }
}
function formatDateToArventoDateString(date) {
    const pad = (num) => num.toString().padStart(2, "0");
    const month = pad(date.getMonth() + 1);
    const day = pad(date.getDate());
    const year = date.getFullYear();
    const hours = pad(date.getHours());
    const minutes = pad(date.getMinutes());
    const seconds = pad(date.getSeconds());
    return `${month}${day}${year}${hours}${minutes}${seconds}`;
}
exports.formatDateToArventoDateString = formatDateToArventoDateString;
